(ns codescene.features.pm-data.trello.trello-fetcher
  "Uses the Trello Rest API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [clj-time.coerce :as c]
            [codescene.cache.core :as cache]
            [codescene.features.pm-data.trello.trello-api :as trello-api]
            [codescene.pm-data.provider-common :as common]
            [evolutionary-metrics.trends.dates :as dates]
            [taoensso.timbre :as log]))

(defn- labels [card]
  (->> (:labels card)
       (mapv :name)))

(defn- stage-after [action]
  (or (get-in action [:data :list :name])
      (get-in action [:data :listAfter :name])))

(defn- action->response [{:keys [id date] :as action}]
  (let [state-after (get-in action [:data :listAfter])
        closed      (get-in action [:data :card :closed])]
    {:id     id
     :action (cond
               state-after :moved 
               closed      :closed
               :else       :unknown)
     :card   (str (get-in action [:data :card :idShort]))
     :state  (stage-after action)
     :date   (common/drop-ms-from-time-string date)}))

(defn- card->response
  [provider-def ->list-name card]
  (let [{:keys [name idShort shortUrl idList created dateLastActivity]} card
        {:keys [supported-work-types]} provider-def]
    {:id          (str idShort)
     :href        shortUrl
     :task-name   name
     :created     (dates/date-time->string (c/from-long created))
     :updated     dateLastActivity
     :status      (->list-name idList)
     :cost        0
     :work-types  (filterv (set supported-work-types) (labels card))}))

(defn fetch-labels [api-client board-ids]
  (->> board-ids
       (mapcat #(trello-api/fetch-labels api-client %))
       (map :name)
       distinct))

(defn fetch-lists [api-client board-ids]
  (->> board-ids
       (mapcat #(trello-api/fetch-lists api-client %))
       (map :name)
       distinct))

(defn make-list-name-lookup [api-client board-id]
  (->> (trello-api/fetch-lists api-client board-id)
       (map (juxt :id :name))
       (into {})))

(defn fetch-boards 
  [api-client]
  (log/infof "Fetch boards")
  (->> (trello-api/fetch-boards api-client)
       (map (juxt :name :id))
       (into {})))

(defn fetch-actions
  [since api-client board-id]
  (log/infof "Fetch actions from Trello since %s" (or since "-"))
  (let [search-options {:since since}]
    (->> (trello-api/fetch-actions api-client board-id search-options)
         (mapv action->response))))

(defn fetch-cards
  [since api-client board-id provider-def]
  (log/infof "Fetch cards from Trello since %s" (or since "-"))
  (let [->list-name (make-list-name-lookup api-client board-id)
        search-options {:since since}]
    (->> (trello-api/fetch-cards api-client board-id search-options)
         (mapv #(card->response provider-def ->list-name %)))))

(def fetch-me trello-api/fetch-me)

(cache/memo-scoped #'fetch-labels)
(cache/memo-scoped #'fetch-lists)
(cache/memo-scoped #'fetch-boards)

(comment
  (def api-client (trello-api/trello-auth (System/getenv "TRELLO_API_KEY") (System/getenv "TRELLO_API_TOKEN")))
  (def board-id "6045ea5748782b336a1feafb")
  (def since (evolutionary-metrics.trends.dates/string->date "2022-03-01"))
  (fetch-actions since api-client board-id)
  (fetch-cards since api-client board-id
               {:rename-work-types {}
                :supported-work-types ["Cloud"]})
  (fetch-labels api-client [board-id])
  (fetch-lists api-client [board-id])
  (make-list-name-lookup api-client board-id)
  (fetch-boards api-client)
  )
