
(ns codescene.features.pm-data.trello.trello-api
  "Contains methods for fetching data over the Trello Rest API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require [clj-http.client :as client]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]
            [taoensso.timbre :as log]))

(def trello-api-url "https://api.trello.com/1")
(def ^:const default-http-page-size 100)
(def ^:const default-http-timeout 30000)

(defn trello-auth
  [api-key api-token]
  (reify api-client/ApiClient
    (augment-request [_this req] (meta-merge req
                                             {:query-params {:key api-key
                                                             :token api-token}}))
    (on-expired-token [_this] nil)))

(defn- creation-ts
  "Extract creation timestamp from the MongoDB id of an entity
   This is necessary since trello does not provide a creation date field on entities.
   See https://help.trello.com/article/759-getting-the-time-a-card-or-board-was-created"
  [entity]
  (->> (subs (:id entity) 0 8)
       (#(Long/parseLong % 16))
       (* 1000)))

(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [api-client method url request-params]
  (let [params (meta-merge
                {:url url
                 :content-type :json
                 :accept       :json
                 :as           :json-strict
                 :request-method method}
                request-params)
        final-params (-> api-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url trello-api-url))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch Trello data from " (:url final-params))
      (client/request final-params))))

(defn api-request
  [api-client method url request-params]
  (:body (api-request* api-client method url request-params)))

(defn get-data [api-client url query-params]
  (api-request api-client :get url
              {:query-params   query-params}))

(defn- get-paged-data [api-client url query-params]
 (let [page-size default-http-page-size]
    (loop [before nil
           total-data []]
      (api-client/check-interrupt)
      (log/infof "Fetching trello data from %s before %s..." url (or before "(none)"))
      (let [query-params (merge query-params
                                {:memberCreator false
                                 :limit page-size
                                 :before before})
            data (map #(assoc % :created (creation-ts %))
                      (get-data api-client url query-params))
            earliest-id (when (seq data)
                          (:id (apply min-key :created data)))
            total-data (concat data total-data)]
        (if (< (count data) page-size)
          total-data
          (recur earliest-id total-data))))))

(defn search [api-client board-id]
  (let [sub-url "/search"
        query (format "board:\"%s\"" board-id)
        page-size default-http-page-size]
    (loop [page 0
           total-cards []]
      (api-client/check-interrupt)
      (log/infof "Fetching trello cards page %d..." page)
      (let [query-params {:query query
                          :card_list true
                          :cards_limit page-size
                          :cards_page page}
            {:keys [cards]} (get-data api-client sub-url query-params)
            total-cards (concat cards total-cards)]
        (if (< (count cards) page-size)
          total-cards
          (recur (inc page) total-cards))))))

(defn fetch-actions [api-client board-id search-options]
 (let [{:keys [since]} search-options
       sub-url (format "/board/%s/actions" board-id)
       query-params (merge {:filter "updateCard:idList,updateCard:closed"
                            :fields "data,date"
                            :memberCreator false}
                           (when since {:since (dates/date-time->string since)}))]
   (get-paged-data api-client sub-url query-params)))

(defn fetch-cards [api-client board-id search-options]
 (let [{:keys [since]} search-options
       sub-url (format "/board/%s/cards/all" board-id)
       query-params (merge {:fields "name,idShort,shortUrl,labels,idList,dateLastActivity"}
                           (when since {:modifiedSince (dates/date-time->string since)}))]
   (get-paged-data api-client sub-url query-params)))

(defn fetch-board [api-client board-id]
  (let [sub-url  (format "/boards/%s" board-id)]
    (get-data api-client sub-url {})))

(defn fetch-labels [api-client board-id]
  (let [sub-url  (format "/boards/%s/labels" board-id)]
    (get-data api-client sub-url {})))

(defn fetch-lists [api-client board-id]
  (let [sub-url  (format "/boards/%s/lists" board-id)]
    (get-data api-client sub-url {})))

(defn fetch-boards [api-client]
    (get-data api-client  "/members/me/boards" {}))

(defn fetch-me [api-client]
  (get-data api-client  "/members/me" {}))

(comment
  (def api-client (trello-auth (System/getenv "TRELLO_API_KEY") (System/getenv "TRELLO_API_TOKEN")))
  (def board-name "CodeScene")
  (def board-id "6045ea5748782b336a1feafb")
  (map :dateLastActivity  (fetch-cards api-client board-id {:since (dates/date-time-string->date "2022-01-25T03:16:53Z")}))
  (first (search api-client board-id))
  (map (juxt :name :id)(fetch-boards  api-client))
  (count (fetch-actions  api-client board-id {:since (dates/string->date "2021-11-01")}))
  (fetch-labels api-client board-id)
  (fetch-lists api-client board-id)
  (fetch-board api-client board-id)
  (fetch-me api-client))
