(ns codescene.features.delta.file-regions
  (:import (java.util NavigableMap TreeMap Comparator)))

(defrecord DiffRegions [starts ends])

(defn diff->regions [start-key count-key diff]
  (let [s (TreeMap. ^Comparator compare)
        e (TreeMap. ^Comparator compare)]
    (doseq [diff-item diff
            :let [start (start-key diff-item)]
            :when start
            :let [end (+ start (max (dec (count-key diff-item)) 0))
                  data (assoc diff-item :start start :end end :non-empty? (pos-int? (count-key diff-item)))]]
      (.put s start data)
      (.put e end data))
    (->DiffRegions s e)))

(defn adjusted-span
  "Returns map of adjusted start and end positions and the two regions that are
  the first region that starts in interval and the last region to end in interval.

  If inclusive is true, also match if start/end indices are inside a region,
  thus returning first region that ends in interval and last region that starts in interval.

  Returns a span start-end with extra metadata."
  [diff-regions start end inclusive?]
  (if diff-regions
    (let [^NavigableMap start-map (if inclusive? (:ends diff-regions) (:starts diff-regions))
          ^NavigableMap end-map (if inclusive? (:starts diff-regions) (:ends diff-regions))
          region-after-start (when-let [[_ v] (.ceilingEntry start-map start)]
                               (when (<= (:start v) end) v))
          region-before-end (when-let [[_ v] (.floorEntry end-map end)]
                              (when (>= (:end v) start) v))]
      {:start (min end (max start (:start region-after-start 0)))
       :end (max start (min end (:end region-before-end Long/MAX_VALUE)))
       :start-region region-after-start
       :end-region region-before-end})
    {:start start :end end}))

(defn proportional-index
  "Using a diff-item, take provided idx and get relative position in before, and return
  same position in after region. If count-before is 0 or 1 it will choose lower bound, unless
  upper? is true."
  [{:keys [start-line-before count-before start-line count]} idx upper?]
  ;; this is super complicated because our end lines are inclusive instead of exclusive
  (if (<= idx start-line-before)
    (cond-> start-line (and upper? (<= count-before 1)) (+ (max (dec count) 0)))
    (let [factor (/ (- idx start-line-before) (max (dec count-before) 1))]
      (if (>= factor 1)
        (+ start-line (max (dec count) 0))
        (+ start-line (long (* factor (max (dec count) 0))))))))

(defn before-to-after-mapping
  "Takes before regions and start and end indices, map them to after indices.

   First adjusted is determined. If a adjusted span is based on actual diff region, we use its after lines."
  [before-regions start end]
  (let [{:keys [start-region end-region] :as res} (adjusted-span before-regions start end true)]
    (when (and start-region end-region)
      (assoc res
        :start (proportional-index start-region start false)
        :end (proportional-index end-region end true)
        :after? true))))

(defn rated-span
  "Rates span for sorting, since each location produced 1-2 spans potentially."
  [{:keys [start end start-region end-region] :as span} after? function]
  (when span
    ;; since sort will make it ascending, lower number is better, we like locations where
    ;; they border or contain the diff, preferring those that don't have borders inside
    ;; diffs, but rather contain diff wholly
    (assoc span :after? after?
                :function function
                :rating (->> [start-region
                              end-region
                              (<= start (:start start-region -1))
                              (<= (:end end-region Long/MAX_VALUE) end)]
                             (map #(if % 1 0))
                             (reduce - 0)))))

(defn change-locator
  "Returns a function that when given change-details it will return a start-end adjusted span: start and end line for
  the change + regions used to determine that. Because Change can have multiple locations, the return is a sorted collection of spans.

  Generally picking first is fine, but this also enables further filtering by the caller.

  If after-indices-only is true then line numbers are always given in after state if possible."
  [diff after-indices-only?]
  (let [regions (diff->regions :start-line :count diff)
        regions-before (diff->regions :start-line-before :count-before diff)]
    (fn [{:keys [locations] :as _change-detail}]
      (->> locations
           (mapcat (fn [{:keys [start-line end-line start-line-before end-line-before function]}]
                     ;; finish with after? true/false
                     [(when start-line (rated-span (adjusted-span regions start-line end-line true) true function))
                      (when start-line-before (rated-span
                                                (if after-indices-only?
                                                  (before-to-after-mapping regions-before start-line-before end-line-before)
                                                  (adjusted-span regions-before start-line-before end-line-before true))
                                                after-indices-only?
                                                function))]))
           (keep identity)
           (sort-by :rating)))))

(defn indices-with-non-empty-region
  "From a collection of spans from change locator, get the first one that has a non-empty region.

  Returns the span, with :region key."
  [spans]
  (some (fn [{:keys [start-region end-region] :as span}]
          (when-let [region (first (filter :non-empty? [start-region end-region]))]
            (assoc span :region region)))
        spans))
