(ns codescene.features.components.chat
  "Chat component protocol for LLM-powered chat assistant.
   
   This component abstracts the chat functionality, allowing different
   implementations for cloud (managed LLM) and on-prem (BYOK - Bring Your Own Key)
   deployments."
  (:require [clojure.spec.alpha :as s]
            [codescene.features.components.core :refer [find-component]]))

(defprotocol ChatComponent
  (-send-message
    [this conversation-id messages options]
    "Send messages to the LLM and get a response. Returns a map with :response and :conversation-id.
     
     Arguments:
       - conversation-id: Optional ID for continuing a conversation (nil for new)
       - messages: Vector of message maps with :role (:user, :assistant, :system) and :content
       - options: Map containing:
         - :project-id - Optional project context
         - :llm-config - LLM configuration (provider, model, api-key for BYOK)
         - :stream? - Whether to stream the response
         - :tools - Available MCP tools
     
     Returns:
       {:conversation-id string
        :response {:role :assistant :content string}
        :tool-calls [...] ; optional, if LLM wants to call tools
        :usage {:prompt-tokens int :completion-tokens int}}"))

(s/def ::role #{:user :assistant :system})
(s/def ::content string?)
(s/def ::message (s/keys :req-un [::role ::content]))
(s/def ::messages (s/coll-of ::message))

(s/def ::provider #{:openai :anthropic :azure :gemini})
(s/def ::model string?)
(s/def ::api-key string?)
(s/def ::llm-config (s/keys :req-un [::provider ::model]
                            :opt-un [::api-key]))

(s/def ::project-id pos-int?)
(s/def ::stream? boolean?)
(s/def ::options (s/keys :opt-un [::project-id ::llm-config ::stream?]))

(s/def ::conversation-id string?)

(defn component [system]
  (find-component system :codescene.features.components/chat))

(s/fdef send-message
  :args (s/cat :this some?
               :conversation-id (s/nilable ::conversation-id)
               :messages ::messages
               :options ::options)
  :ret map?)

(defn send-message
  "Send messages to the LLM and get a response."
  [this conversation-id messages options]
  (-send-message this conversation-id messages options))
