(ns codescene.features.chat.service.prompt.tools
  "Tool description formatting for system prompts.
   
   Builds human-readable tool descriptions to include in LLM system prompts."
  (:require [clojure.string :as str]))

(def ^:private response-format-instructions
  "Instructions for LLM on how to format structured responses."
  (str "\n\n## Response Format\n\n"
       "You MUST respond with valid JSON in this exact format:\n\n"
       "```json\n"
       "{\n"
       "  \"thinking\": \"Your internal reasoning about the request (optional)\",\n"
       "  \"tool_call\": null or {\"name\": \"tool_name\", \"arguments\": {...}},\n"
       "  \"response\": \"Your response to the user (required if no tool_call)\"\n"
       "}\n"
       "```\n\n"
       "Rules:\n"
       "1. If you need to use a tool, set tool_call with name and arguments, set response to null\n"
       "2. If you can answer directly without tools, set tool_call to null and provide your response\n"
       "3. After receiving tool results, incorporate them into your response\n"
       "4. Always respond with valid JSON only - no markdown code blocks around it\n"
       "5. The response field should contain your final answer in markdown format when appropriate"))

(defn- required-marker
  "Returns '(required)' or '(optional)' based on whether param is in required set."
  [param-name required-set]
  (if (required-set param-name)
    " (required)"
    " (optional)"))

(defn- format-param-description
  "Formats the optional description suffix for a parameter."
  [description]
  (when description
    (str " - " description)))

(defn- format-single-parameter
  "Formats a single parameter as a readable line."
  [{:keys [name type description required-set]}]
  (format "    - %s: %s%s%s"
          name
          (or type "any")
          (required-marker name required-set)
          (or (format-param-description description) "")))

(defn- property->param-ctx
  "Converts a property map entry to parameter context for formatting."
  [required-set [k v]]
  {:name (clojure.core/name k)
   :type (:type v)
   :description (:description v)
   :required-set required-set})

(defn format-tool-parameters
  "Formats tool parameters as a readable description."
  [parameters]
  (when-let [props (:properties parameters)]
    (let [required-set (set (:required parameters []))]
      (->> props
           (map (partial property->param-ctx required-set))
           (map format-single-parameter)
           (str/join "\n")))))

(defn- format-single-tool
  "Formats a single tool definition as markdown."
  [{:keys [name description parameters]}]
  (str "### " name "\n"
       description "\n"
       (when-let [params (format-tool-parameters parameters)]
         (str "Parameters:\n" params "\n"))))

(defn build-tools-system-prompt
  "Builds a system prompt section describing available tools."
  [tools]
  (if (empty? tools)
    ""
    (str "\n\n## Available Tools\n\n"
         "You have access to the following tools. To use a tool, include a tool_call in your response.\n\n"
         (->> tools (map format-single-tool) (str/join "\n")))))

(defn build-system-prompt-with-tools
  "Builds a complete system prompt including tool descriptions."
  [base-system-prompt tools]
  (str base-system-prompt
       (build-tools-system-prompt tools)
       response-format-instructions))
