(ns codescene.features.chat.mcp-servers.insights
  "Insights MCP server implementation."
  (:require [clojure.string :as str]
            [codescene.features.chat.service.mcp.protocol :as protocol]
            [codescene.features.chat.service.mcp.server :refer [MCPServer]]
            [codescene.features.chat.service.mcp.session :as session]
            [taoensso.timbre :as log]))

(defn- build-headers
  "Builds headers for MCP requests."
  [{:keys [auth-token instance-url config]}]
  (cond-> {}
    auth-token (assoc "Authorization" (str "Bearer " auth-token))
    instance-url (assoc "x-cs-instance-url" instance-url)
    (:headers config) (merge (:headers config))))

(defn- build-request-context
  "Builds the request context with server-specific headers merged in."
  [{:keys [auth-token http-opts mcp-url] :as context}]
  (let [headers (build-headers context)
        session-id (session/get-session auth-token)]
    (cond-> {:url (or mcp-url (session/get-mcp-url))
             :session-id session-id
             :headers headers
             :on-new-session #(session/set-session! auth-token %)
             :on-reset-session #(session/reset-session! auth-token)}
      http-opts (assoc :http-opts http-opts))))

(defn- convert-tool [tool]
  {:name (:name tool)
   :description (:description tool)
   :parameters (or (:inputSchema tool) {:type "object" :properties {}})})

(defn- convert-prompt [prompt]
  {:name (:name prompt)
   :title (:title prompt)
   :description (:description prompt)
   :arguments (or (:arguments prompt) [])})

(defn- fetch-tools [context]
  (let [req-ctx (build-request-context context)
        response (protocol/send-request-with-retry req-ctx "tools/list" {})]
    (if (:success response)
      {:success true :tools (get-in response [:result :tools] [])}
      {:success false :error (:error response)})))

(defn- fetch-prompts [context]
  (let [req-ctx (build-request-context context)
        response (protocol/send-request-with-retry req-ctx "prompts/list" {})]
    (if (:success response)
      {:success true :prompts (get-in response [:result :prompts] [])}
      {:success false :error (:error response)})))

(defn- get-prompt-content [{:keys [prompt-name arguments] :as context}]
  (let [req-ctx (build-request-context context)]
    (protocol/send-request-with-retry req-ctx "prompts/get" {:name prompt-name :arguments (or arguments {})})))

(defn- extract-text-content [content]
  (->> content
       (filter #(= (:type %) "text"))
       (map :text)
       (str/join "\n")))

(defn- log-tool-call [{:keys [tool-name arguments auth-token instance-url]}]
  (log/infof "Executing Insights MCP tool: %s with arguments: %s (auth-token present: %s, instance-url: %s)"
             tool-name arguments (boolean auth-token) instance-url))

(defn- log-tool-success [tool-name text-content]
  (log/infof "Insights MCP tool %s succeeded, result length: %d chars"
             tool-name (count (str text-content))))

(defn- log-tool-failure [tool-name error]
  (log/warnf "Insights MCP tool %s failed: %s" tool-name error))

(defn- build-success-result [response]
  (let [content (get-in response [:result :content] [])
        text-content (extract-text-content content)]
    {:success true
     :result (if (empty? text-content)
               (:result response)
               text-content)
     :text-content text-content}))

(defn- call-tool [{:keys [tool-name arguments] :as context}]
  (let [req-ctx (build-request-context context)]
    (protocol/send-request-with-retry req-ctx "tools/call" {:name tool-name :arguments (or arguments {})})))

(defn- connection-error-message
  "Formats a user-friendly error message for connection failures."
  [error]
  (cond
    (str/includes? error "Connection refused")
    (str "Could not connect to CodeScene instance. "
         "Please verify that your instance URL is correct and the server is running.")
    
    (str/includes? error "UnknownHostException")
    (str "Could not resolve CodeScene instance hostname. "
         "Please check that your instance URL is correct.")
    
    (str/includes? error "timeout")
    (str "Connection to CodeScene instance timed out. "
         "Please check your network connection and try again.")
    
    :else
    (str "Failed to connect to CodeScene instance: " error)))

(defrecord InsightsMCPServer []
  MCPServer
  
  (server-name [_]
    :insights)
  
  (list-tools [_ context]
    (protocol/ensure-initialized (build-request-context context))
    (let [result (fetch-tools context)]
      (if (:success result)
        (do
          (log/infof "Insights MCP returned %d tools" (count (:tools result)))
          {:tools (mapv convert-tool (:tools result))
           :errors []})
        (do
          (log/warnf "Failed to list Insights MCP tools: %s" (:error result))
          {:tools []
           :errors [(connection-error-message (:error result))]}))))
  
  (list-prompts [_ context]
    (protocol/ensure-initialized (build-request-context context))
    (let [result (fetch-prompts context)]
      (if (:success result)
        (do
          (log/infof "Insights MCP returned %d prompts" (count (:prompts result)))
          {:prompts (mapv convert-prompt (:prompts result))
           :errors []})
        (do
          (log/warnf "Failed to list Insights MCP prompts: %s" (:error result))
          {:prompts []
           :errors [(connection-error-message (:error result))]}))))
  
  (get-prompt [_ context]
    (protocol/ensure-initialized (build-request-context context))
    (log/infof "Getting Insights MCP prompt: %s" (:prompt-name context))
    (let [response (get-prompt-content context)]
      (if (:success response)
        (do
          (log/infof "Insights MCP prompt %s retrieved successfully" (:prompt-name context))
          {:success true :result (:result response)})
        (do
          (log/warnf "Failed to get Insights MCP prompt %s: %s" (:prompt-name context) (:error response))
          {:success false :error (:error response)}))))
  
  (execute-tool [_this context]
    (protocol/ensure-initialized (build-request-context context))
    (log-tool-call context)
    (let [response (call-tool context)]
      (log/debugf "CodeScene MCP tool response: %s" response)
      (if (:success response)
        (let [result (build-success-result response)]
          (log-tool-success (:tool-name context) (:text-content result))
          (dissoc result :text-content))
        (do
          (log-tool-failure (:tool-name context) (:error response))
          {:success false :error (:error response)})))))

(defn create-server []
  (->InsightsMCPServer))

(defn reset-session! [auth-token]
  (session/reset-session! auth-token))
