(ns codescene.features.api.safeguards
  "REST API endpoints for Code Health safeguards.
   Provides insights and outcomes for Code Health quality gates from various sources.
   Currently supports PR-based safeguards, extensible to MCP and IDE sources."
  (:require [clojure.string :as str]
            [codescene.features.api.core :as api-core]
            [codescene.features.components.analysis :as analysis]
            [codescene.features.util.api :as api-utils]
            [codescene.util.date :as date-utils]
            [codescene.analysis.paths :as paths]
            [codescene.util.json :as json]
            [codescene.delta.statistics.trends :as trends]
            [clj-time.core :as t]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- trend-in-date-range?
  "Check if a monthly trend falls within the date range"
  [{:keys [date]} from-date to-date]
  (when-let [trend-date (trends/yy-mm->date date)]
    (and (not (t/before? trend-date from-date))
         (not (t/after? trend-date to-date)))))

(defn- interpret-trend
  "Interpret the trend data to provide actionable insights.
   net-change equals the ignored findings count — how many findings were
   merged without being fixed or suppressed.
   Status: :no-issues (nothing detected), :stable (findings detected but
   none ignored), :degrading (some findings merged unaddressed)."
  [{:keys [total-negative-findings total-ignored-findings]
    :or {total-negative-findings 0
         total-ignored-findings 0}}]
  {:net-change total-ignored-findings
   :status (cond
             (zero? total-negative-findings) :no-issues
             (zero? total-ignored-findings) :stable
             :else :degrading)})

(defn- summarize-trends
  "Summarize multiple monthly trends into aggregate statistics"
  [trends]
  (let [summaries (map :summary trends)
        sum-field (fn [field]
                    (reduce + 0 (map #(get % field 0) summaries)))]
    {:total-delta-analyses (sum-field :total-delta-analyses)
     :total-detected-violations (sum-field :total-detected-violations)
     :total-detected-violations-fixed (sum-field :total-detected-violations-fixed)
     :total-violations-ignored (sum-field :total-violations-ignored)
     :total-negative-findings (sum-field :total-negative-findings)
     :total-ignored-findings (sum-field :total-ignored-findings)
     :total-suppressed-findings (sum-field :total-suppressed-findings)}))

(defn- pr-trends->insights
  "Transform PR statistics trends into insights response"
  [trends meta-info]
  (let [summary (summarize-trends trends)
        interpretation (interpret-trend summary)
        detected (:total-negative-findings summary)
        ignored (:total-ignored-findings summary)
        suppressed (:total-suppressed-findings summary)
        fixed (- detected ignored suppressed)]
    {:meta meta-info
     :summary {:total-pr-analyses (:total-delta-analyses summary)
               :findings-detected detected
               :findings-fixed fixed
               :findings-ignored ignored
               :findings-suppressed suppressed}
     :interpretation interpretation
     :monthly-trends (mapv (fn [{:keys [date summary]}]
                             {:month date
                              :statistics summary
                              :interpretation (interpret-trend summary)})
                           trends)}))

(defn- pr-stat-to-outcome
  "Convert a PR statistics entry to an outcome"
  [pr-stat]
  (let [{:keys [external-review-id last-pr-analysis-date repo authors
                total-negative-findings
                ignored-findings suppressed-findings]} pr-stat
        detected total-negative-findings
        ignored ignored-findings
        suppressed suppressed-findings
        fixed (- detected ignored suppressed)]
    {:pr-id external-review-id
     :analysis-date last-pr-analysis-date
     :repository repo
     :authors authors
     :statistics {:findings-detected detected
                  :findings-fixed fixed
                  :findings-ignored ignored
                  :findings-suppressed suppressed}
     :interpretation {:net-change ignored
                      :status (cond
                                (zero? detected) :no-issues
                                (zero? ignored) :resolved
                                :else :unresolved)}}))

(defn- pr-in-date-range?
  "Check if a PR's last analysis date falls within the date range"
  [{:keys [last-pr-analysis-date]} from-str to-str]
  (when last-pr-analysis-date
    (try
      (let [pr-date-str (subs last-pr-analysis-date 0 10)] ;; Extract YYYY-MM-DD from ISO string
        (and (>= (compare pr-date-str from-str) 0)
             (<= (compare pr-date-str to-str) 0)))
      (catch Exception _
        false))))

(defn- empty-insights-response
  [meta-info]
  {:meta meta-info
   :summary {:total-pr-analyses 0
             :findings-detected 0
             :findings-fixed 0
             :findings-ignored 0
             :findings-suppressed 0}
   :interpretation {:net-change 0
                    :status :no-issues}
   :monthly-trends []})

(defn- insights-from-analysis
  [{:keys [analysis-component project-id analysis-id from-date to-date meta-info]}]
  (let [path-fn (analysis/local-analysis-result-path analysis-component project-id analysis-id)
        trends-file (path-fn paths/delta-trend-results-json)
        trends (let [data (json/read-when-exists trends-file)]
                 (if (seq data) data []))
        filtered-trends (filter #(trend-in-date-range? % from-date to-date) trends)]
    (pr-trends->insights filtered-trends meta-info)))

(defn insights
  "Get aggregated insights for code health quality gates for a project within a date range.
   This reads the PR statistics trends data (delta_trend_results.json) from the latest analysis."
  [system {:keys [project-id from to]}]
  (let [generated-at (t/now)
        from-date-str (date-utils/get-date-string from (t/minus generated-at (t/months 1)))
        to-date-str (date-utils/get-date-string to generated-at)
        from-date (dates/string->date from-date-str)
        to-date (dates/string->date to-date-str)
        analysis-component (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis analysis-component project-id)
        meta-info {:project-id project-id
                   :from from-date-str
                   :to to-date-str
                   :generated-at generated-at}]
    (api-utils/ok
     (if latest-analysis
       (insights-from-analysis {:analysis-component analysis-component
                                :project-id project-id
                                :analysis-id (:id latest-analysis)
                                :from-date from-date
                                :to-date to-date
                                :meta-info (assoc meta-info :analysis-id (:id latest-analysis))})
       (empty-insights-response meta-info)))))

(defn- get-pr-stats
  "get detailed pr statistics filtering out records without authors
   see: https://app.clickup.com/t/9015696197/CS-4556"
  [analysis-component project-id analysis-id]
  (let [path-fn (analysis/local-analysis-result-path analysis-component project-id analysis-id)
        pr-stats-file (path-fn paths/delta-detailed-pr-statistics-result-json)
        pr-stats (let [data (json/read-when-exists pr-stats-file)]
                   (if (seq data) data []))]
    (filter #(-> % :authors not-empty) pr-stats)))

(defn- outcomes-from-analysis
  [{:keys [analysis-component project-id analysis-id from-date-str to-date-str]}]
  (let [pr-stats (get-pr-stats analysis-component project-id analysis-id)
        filtered-prs (filter #(pr-in-date-range? % from-date-str to-date-str) pr-stats)]
    (mapv pr-stat-to-outcome filtered-prs)))

(defn outcomes
  "Get individual PR outcomes for code health quality gates for a project within a date range.
   This reads the detailed PR statistics (delta_detailed_pr_statistics_result.json) from the latest analysis."
  [system {:keys [project-id from to page page-size]}]
  (let [from-date-str (date-utils/get-date-string from (t/minus (t/now) (t/months 1)))
        to-date-str (date-utils/get-date-string to (t/now))
        analysis-component (api-core/api-analyses system)
        latest-analysis (analysis/latest-successful-analysis analysis-component project-id)
        results (if latest-analysis
                  (outcomes-from-analysis {:analysis-component analysis-component
                                           :project-id project-id
                                           :analysis-id (:id latest-analysis)
                                           :from-date-str from-date-str
                                           :to-date-str to-date-str})
                  [])]
    (api-utils/paginated-list-result results
                                     {:page page
                                      :page-limit page-size
                                      :list-kw :outcomes
                                      :processing-fn identity})))

(defn outcome-pr->csv
  "get the pull request details to be converted as CSV"
  [system project-id analysis-id]
  (let [header ["Pull Request" "Date" "Detected" "Fixed" "Ignored" "Suppressed" "Authors"]
        analysis-component (api-core/api-analyses system)
        pr-stats (get-pr-stats analysis-component project-id analysis-id)
        ->row (fn [m] (let [{pull_request :external-review-id date :last-pr-analysis-date
                             detected :total-negative-findings ignored :ignored-findings
                             suppressed :suppressed-findings authors :authors} m
                             fixed (- detected ignored suppressed)]
                        [pull_request date detected fixed ignored suppressed (str/join ";" authors)]))]
    (concat [header] (map ->row pr-stats))))
