(ns codescene.features.api.analytical
  (:require [codescene.features.analysis.util :as analysis-util]
            [codescene.features.tag-queries.core :as tag-queries]
            [codescene.features.components.db :as db]
            [codescene.features.util.api :as api-utils]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(defn- paginated-data
  [system data-fn results-k {:keys [page filter-str] :as _options}]
  (let [db-spec (db/db-spec system) ]
    (-> (data-fn db-spec)
        (analysis-util/paginated-ordered-data {:page page
                                               :page-size api-utils/default-page-limit
                                               :filter-str filter-str
                                               :results-k results-k})
        api-utils/ok)))

(defn- with-exceptions-as-bad-request*
  [body-fn]
  (try+
    (body-fn)
    (catch [] e
      (log/error (:throwable &throw-context) (:message &throw-context))
      (api-utils/bad-request (:message &throw-context)))))

(defmacro with-exceptions-as-bad-request
  [& body]
  `(with-exceptions-as-bad-request* (fn [] ~@body)))

(defn analytical-tags
  [system _request options]
  (let [data-fn (fn [db-spec]
                  (->> (tag-queries/analytical-tags db-spec)
                       (map #(select-keys % [:id :name]))))]
    (paginated-data system data-fn :tags options)))

(defn add-tag
  [system _request {:keys [name] :as _body}]
  (with-exceptions-as-bad-request
   (->> {:id (tag-queries/add-analytical-tag (db/db-spec system) name)}
        api-utils/created)))

(defn delete-tag
  [system _request tag-id]
  (-> {:success (tag-queries/delete-analytical-tag (db/db-spec system) tag-id)}
      api-utils/ok))

(defn assign-tag
  [system _request {:keys [tag-id analytical-unit-id] :as _body}]
  (with-exceptions-as-bad-request
    (-> {:success (tag-queries/assign-analytical-tag (db/db-spec system) tag-id analytical-unit-id)}
      api-utils/ok)))

(defn unassign-tag
  [system _request {:keys [tag-id analytical-unit-id] :as _body}]
  (with-exceptions-as-bad-request
    (-> {:success (tag-queries/unassign-analytical-tag (db/db-spec system) tag-id analytical-unit-id)}
      api-utils/ok)))

(defn tag-assignments
  [system _request options]
  (let [data-fn (fn [db-spec]
                  (->> (tag-queries/analytical-tag-assignments db-spec)
                       (map #(select-keys % [:analytical_unit_id :tag_id]))))]
    (paginated-data system data-fn :assignments options)))

(defn analytical-units
  [system _request options]
  (let [data-fn (fn [db-spec]
                  (->> (tag-queries/analytical-units db-spec)
                       (map #(select-keys % [:id :name :type :project-id]))))]
    (paginated-data system data-fn :units options)))

(defn analytical-units-with-tags
  [system _request {:keys [types tag-ids] :as _body} options]
  (let [data-fn (fn [db-spec]
                  (->> (tag-queries/analytical-units-with-tags db-spec types tag-ids)
                       (map #(select-keys % [:id :name :type :project-id]))))]
    (paginated-data system data-fn :units options)))

