(ns codescene.features.analysis-queries.loading
  (:require
   [clojure.java.jdbc :as jdbc]
   [codescene.features.util.maps :refer [->kebab]]
   [hugsql.core :as hugsql])
  (:import (java.sql SQLException)))

(hugsql/def-db-fns "codescene/features/analysis_queries/queries.sql")

(defn make-analytical-unit-type-lookup
  [db-spec]
  (->> (select-all-analytical-unit-types db-spec {})
       (map (fn [{:keys [id type_name]}]
              [(keyword type_name) id]))
       (into {})))

(defn make-metric-type-lookup-table
  [db-spec]
  (->> (select-all-metric-types db-spec {})
       (map (fn [{:keys [id metric_name]}]
              [(keyword metric_name) id]))
       (into {})))

(defn analytical-unit->readable
  [au]
  (-> au ->kebab))

(defn observation->readable
  [ob]
  (-> ob ->kebab (update :metric-name keyword)))

(defn- undo-delete-analytical-unit
  "Returns the unit after update."
  [db-spec unit-id]
  (jdbc/with-db-transaction [tx db-spec]
    (set-to-not-deleted! tx {:id unit-id})
    (-> (select-analytical-unit-by-id tx {:id unit-id})
        analytical-unit->readable)))

(defn delete-analytical-unit
  "Marks an analytical unit as deleted by setting deleted_at to current timestamp."
  [db-spec unit-id]
  (jdbc/with-db-transaction [tx db-spec]
    (set-to-deleted! tx {:id unit-id})))

(defn- get-analytical-unit-by-id
  [tx id]
  (-> (select-analytical-unit-by-id tx {:id id})
      analytical-unit->readable))

(defn get-analytical-unit-by-name-and-type
  [{:keys [db-spec account-id project-id]} unit-type unit-name]
  (-> (select-analytical-unit-by-name-and-type db-spec {:name unit-name :unit_type_id unit-type
                                                         :project_id project-id :account_id account-id})
      analytical-unit->readable))

(defn- get-key-after-insertion
  "Tries to bridge the gap between h2, which provides a map with `:id`
  and mysql, which provides a map with `:generated_key`."
  [db-response]
  (or (get db-response :id) (get db-response :generated-key)))

(defn- insert-analytical-unit!
  [db-spec {:keys [unit-type-lookup account-id project-id]} parent-id {unit-type :type unit-name :name}]
  (jdbc/with-db-transaction [tx db-spec]
    (->> (insert-analytical-unit tx {:project_id project-id
                                     :account_id account-id
                                     :unit_type_id (get unit-type-lookup unit-type)
                                     :parent_id parent-id
                                     :name unit-name})
         get-key-after-insertion
         (get-analytical-unit-by-id tx))))

(defn- insert-project!
  [db-spec extraction-context {project-name :name}]
  (insert-analytical-unit! db-spec extraction-context nil {:type :project :name project-name}))

(defn get-or-insert-project
  [{:keys [db-spec unit-type-lookup account-id] :as extraction-context} {:keys [project-id] project-name :name :as project-info}]
  (jdbc/with-db-transaction [tx db-spec]
    (let [select-params {:project_id project-id
                         :account_id account-id
                         :unit_type_id (get unit-type-lookup :project)}
          existing (-> (select-analytical-units-by-project-id-and-type tx select-params)
                       first
                       analytical-unit->readable)]
      (cond
        (and existing (:deleted-at existing))
        (undo-delete-analytical-unit tx (:id existing))

        existing existing

        :else
        (insert-project! db-spec extraction-context project-info)))))

(defn get-or-insert-sub-project-unit
  [{:keys [db-spec unit-type-lookup account-id project-id] :as extraction-context}
   {analytical-unit-parent-id :id :as _project} ; this is the already inserted project
   {unit-name :name unit-type :type :as sub-project-unit}]
  (jdbc/with-db-transaction [tx db-spec]
    (let [unit-type (get unit-type-lookup unit-type)
          ctx (assoc extraction-context :db-spec tx)
          existing (get-analytical-unit-by-name-and-type ctx unit-type unit-name)]
      (cond
        (and existing (:deleted-at existing))
        (undo-delete-analytical-unit tx (:id existing))

        existing existing

        :else
        (insert-analytical-unit! tx extraction-context analytical-unit-parent-id sub-project-unit)))))




(defn insert-observation
  [db-spec metric-type-lookup au-id {:keys [metric value] :as observation}]
  (insert-observation! db-spec {:analytical_unit_id au-id
                                :metric_id (get metric-type-lookup metric)
                                :value value}))

(defn insert-observations-from
  [db-spec metric-type-lookup {:keys [observations id] :as au}]
  (jdbc/with-db-transaction [tx db-spec]
    (doseq [ob observations]
      (insert-observation tx metric-type-lookup id ob))))

(defn get-observations-for-analytical-unit
  "Probably too general of a function for production use, since it would
  fetch large numbers of observations. Using for testing at the
  moment. This functionality will get moved to another ns."
  [db-spec account-id au-id]
  (->>
    (select-observations-for-analytical-unit db-spec {:analytical_unit_id au-id
                                                      :account_id account-id})
    (map observation->readable)))

(defn- project?
  [au]
  (= :project (:type au)))

(defn load-from-analysis
  "Expects the results from 
  `codescene.post-analysis.measurement-model-builder/build-analytical-units-for!`. 
  Loads them into the DB, creating the analytical units when necessary."
  [db-spec account-id analytical-units-from-results]
  (jdbc/with-db-transaction [tx db-spec]
    (let [project (some #(when (project? %) %) analytical-units-from-results)
          metric-type-lookup (make-metric-type-lookup-table tx)
          ctx {:db-spec tx
               :unit-type-lookup  (make-analytical-unit-type-lookup tx)
               :account-id account-id
               :project-id (:project-id project)}
          project-db-au (get-or-insert-project ctx project)
          project-with-id (assoc project :id (:id project-db-au))]

      (insert-observations-from tx metric-type-lookup project-with-id)

      (doseq [au (remove project? analytical-units-from-results)]
        (let [sub-unit-db-au (get-or-insert-sub-project-unit ctx project-db-au au)
              au-with-id (assoc au :id (:id sub-unit-db-au))]
          (insert-observations-from tx metric-type-lookup au-with-id))))))


