(ns codescene.features.analysis.util.technical-debt
  (:require [semantic-csv.core :as sc]
            [clojure.set :as set]
            [medley.core :as m]
            [codescene.analysis.paths :as paths]
            [codescene.presentation.display :as display]
            [codescene.features.util.csv :as csv]))

(defn- ->double
  "Coerce to double, handling scientific notation (e.g. 7.5E-4)."
  [v]
  (cond
    (number? v) (double v)
    (string? v) (try (Double/parseDouble v) (catch Exception _ v))
    :else v))

(defn- td-target-type->str
  [target-type]
  (cond (= "primary" target-type) "recommended-refactoring-target"
        (= "secondary" target-type) "recommended-refactoring-target"
        (= "local" target-type) "local-refactoring-opportunity"
        :else "none"))

(defn- technical-debt-friction
  [path-fn]
  (let [td-refactoring-targets (sc/slurp-csv (path-fn paths/td-refactoring-targets-csv))
        td-refactoring-target-type (fn [file-name]
                                     (some->> td-refactoring-targets
                                              (filter #(= (:file-name %) file-name))
                                              first
                                              :target-type))
        kmap {:file-name :file_name
              :friction-last-month :friction_last_month
              :friction-last-year :friction_last_year}
        cast-to-int (partial reduce #(m/update-existing %1 %2 display/->maybe-int))
        keys-to-cast-to-int [:revisions :loc]
        cast-to-double (partial reduce #(m/update-existing %1 %2 ->double))
        keys-to-cast-to-double [:code_health_score :friction :friction_last_month :friction_last_year]
        transform-fn #(-> %
                          (assoc :code_health_score (:code-health %)
                                 :refactoring_target (-> (td-refactoring-target-type (:file-name %))
                                                         td-target-type->str))
                          (dissoc :code-health)
                          (set/rename-keys kmap)
                          (cast-to-int keys-to-cast-to-int)
                          (cast-to-double keys-to-cast-to-double))]
    (->> (path-fn paths/td-friction-by-file-csv)
         csv/read-csv
         (map transform-fn))))

(defn- technical-debt-hotspots
  [path-fn]
  (let [td-friction-by-file (sc/slurp-csv (path-fn paths/td-friction-by-file-csv))
        td-friction-by-file-val (fn [file-name k]
                                  (some->> td-friction-by-file
                                           (filter #(= (:file-name %) file-name))
                                           first
                                           k))
        kmap {:file-name :file_name}
        cast-to-int (partial reduce #(m/update-existing %1 %2 display/->maybe-int))
        keys-to-cast-to-int [:revisions :loc]
        cast-to-double (partial reduce #(m/update-existing %1 %2 ->double))
        keys-to-cast-to-double [:code_health_score :friction :friction_last_month :friction_last_year]
        transform-fn #(-> %
                          (assoc :code_health_score (td-friction-by-file-val (:file-name %) :code-health)
                                 :loc (td-friction-by-file-val (:file-name %) :loc)
                                 :revisions (td-friction-by-file-val (:file-name %) :revisions)
                                 :friction (td-friction-by-file-val (:file-name %) :friction)
                                 :friction_last_month (td-friction-by-file-val (:file-name %) :friction-last-month)
                                 :friction_last_year (td-friction-by-file-val (:file-name %) :friction-last-year)
                                 :refactoring_target (td-target-type->str (:target-type %)))
                          (dissoc :target-type)
                          (set/rename-keys kmap)
                          (cast-to-int keys-to-cast-to-int)
                          (cast-to-double keys-to-cast-to-double))]
    (->> (path-fn paths/td-refactoring-targets-csv)
         csv/read-csv
         (map transform-fn))))

(defn technical-debt
  [path-fn {:keys [refactoring-targets?]}]
  (if refactoring-targets?
    (technical-debt-hotspots path-fn)
    (technical-debt-friction path-fn)))