(ns codescene.features.repository-provider.gitlab.groups
  (:require [codescene.features.repository-provider.gitlab.api :as api]
            [codescene.features.util.log :as u.log]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(def ^:private group-id #(-> % :group :id))

(defn top-group [project] (re-find #"^[^/]+" (-> project :namespace :full_path)))

(defn from-projects
  "Select all user's projects first, since that endpoint will return
  projects are transitively available via shared groups. Then select group details to figure out
  user's access. Downside here is that top level groups with no projects will not be detected."
  [authed-client]
  (let [{:keys [id username]} (api/get-user authed-client)]
    (u.log/log-excessive-time
      (->> (into (sorted-set)
                 (map top-group)
                 (api/get-user-projects authed-client true))
           (api/get-groups-by-names authed-client)
           (keep (fn [g]
                   (when-let [access-level (api/user-group-access authed-client (:id g) id)]
                     {:group g
                      :role (api/access-level->org-role access-level)})))
           doall)
      (log/logf :warn
                "Fetching all user's groups took too long for GitLab user %s [Elapsed time: %.3f s]" username %t))))

;; I'll keep this if needed later
(defn with-shared-groups 
  "Find all group IDs that can be reached from the initial group of IDs.
  
  We have to expand non-top level groups as well."
  [authed-client]
  (let [initial-groups (api/get-groups authed-client false)]
    (loop [groups (m/index-by group-id initial-groups)
           to-do (into #{} (map group-id) initial-groups)]
      (if-let [id (first to-do)]
        ;; make sure to not add IDs again that we already did and they are in group-ids
        (let [shared-groups (api/shared-groups authed-client id)]
          (recur (merge groups (m/index-by group-id shared-groups))
                 (disj (into to-do (comp (map group-id) (remove groups)) shared-groups)
                       id)))
        (vals groups)))))