(ns codescene.features.reports.code-health.pdf-elements
  (:require
    [codescene.features.reports.pdf-helpers :as ph]
    [codescene.features.reports.common.styling :as styling]
    [codescene.features.reports.common.project-card :as project-card]
    [codescene.features.reports.common.pdf-elements :as pdf-elements]
    [codescene.util.time :as ut]
    [evolutionary-metrics.trends.dates :as dates]))


(defn- project-name-rows [project-name maxlength]
  (let [split-index (min maxlength (count project-name))
        row1 (subs project-name 0 split-index)
        row2-raw (subs project-name split-index)
        row2 (styling/string->cutoff-string row2-raw maxlength)]
    [row1 row2]))

(defn- analysis-time 
  "The analysis metadata contains clj-time, here we expect java.time.Instant..."
  [{:keys [analysis-time] :as _analysis}]
  (-> analysis-time dates/date-time->string ut/instant))

(defn- title-page-bottom-content [{:keys [dashboard file-summary this-period-statistics analysis] :as _data} project-name img-dir]
  (let [[project-name-1 project-name-2] (project-name-rows (or project-name "") 27)
        loc (or (:application-code-loc dashboard)
                (reduce + (map :code file-summary)))
        formatted-loc (styling/number->printable-string loc)
        loc-img (styling/load-resource img-dir "Code.png")
        files (reduce + (map :files file-summary))
        formatted-files (styling/number->printable-string files)
        files-img (styling/load-resource img-dir "File-text.png")
        devs (:contributors this-period-statistics)
        formatted-devs (styling/number->printable-string devs)
        devs-img (styling/load-resource img-dir "Users.png")
        end (analysis-time analysis)
        formatted-lad (styling/instant->printable-string end)
        lad-img (styling/load-resource img-dir "Calendar.png")
        item-col-widths [2 9]]
    (ph/row
     (ph/cell
      []
      [:spacer 4]
      (ph/paragraph [:tp-miniheader] "PROJECT SUMMARY")
      (ph/table
       [] nil
       (ph/row
        (ph/cell
         [:tp-subtitle-cell]
         (ph/paragraph [:tp-subtitle-paragraph] project-name-1)))
       (ph/row
        (ph/cell
         [:tp-subtitle-cell]
         (ph/paragraph [:tp-subtitle-paragraph] project-name-2))))
      [:spacer 3]
      (ph/table
       [:tp-list-table] nil
       (ph/row
        (ph/cell [:tp-list-cell] (pdf-elements/title-page-item item-col-widths loc-img "Lines of code" formatted-loc))
        (ph/cell [:tp-list-cell] (pdf-elements/title-page-item item-col-widths files-img "Number of files" formatted-files)))
       (ph/row
        (ph/cell [:tp-list-cell] (pdf-elements/title-page-item item-col-widths devs-img "Active developers" formatted-devs))
        (ph/cell [:tp-list-cell] (pdf-elements/title-page-item item-col-widths lad-img "Latest analysis results" formatted-lad))))
      [:spacer 4]))))

(defn title-page-contents [data project-name img-dir date]
  (ph/table
   [:tp-top-table] nil
   (ph/row
    (ph/cell
     []
     [:spacer 5]
     (ph/paragraph [:tp-miniheader] "CODESCENE REPORT")
     (ph/paragraph [:ch-tp-title-paragraph] "Code Health Overview Report")
     [:spacer 3]
     (ph/line [:tp-separation-line])))
   (title-page-bottom-content data project-name img-dir)
   (ph/row
    (ph/cell
     []
     (ph/line [:tp-separation-line])
     [:spacer 4]
     (ph/paragraph [:tp-footer1] (format "Generated on:  %s" (styling/instant->printable-string date)))
     [:spacer 1]
     (ph/paragraph [:tp-footer2] "By CodeScene")))))


(defn page1 [{:keys [img-dir]} contentlist]
  [[:svg {:under true} (styling/load-resource img-dir "contentpage-background.svg")]
   (for [content contentlist]
     (ph/table [:c-card-table] nil (ph/row (ph/cell [:c-card-cell :c-card-cell-padding] content))))])


(defn page2 [{:keys [img-dir]} beginning end contentlist]
  (let [interval-string (styling/interval->printable-string beginning end)
        page-header (format "Statistics and Trends for %s" interval-string)]
    [[:svg {:under true} (styling/load-resource img-dir "contentpage-background.svg")]
     (ph/paragraph [:c-page-header-paragraph] page-header)
     (for [content contentlist]
       (ph/table [:c-card-table] nil (ph/row (ph/cell [:c-card-cell :c-card-cell-padding] content))))]))


(defn all-factors [period-renderable-trends {:keys [img-dir dashboard-url]}]
  (ph/table
   [:c-card-content-top-table] nil
   (ph/row
      (ph/cell
        [:ch-factors-header-cell]
        (ph/table
          [] nil
          (ph/row
            (ph/cell [] (ph/paragraph [:c-miniheader] "PROJECT OVERVIEW"))
            (ph/cell [] (ph/table [] [10 1] (pdf-elements/link-row dashboard-url "Go to Dashboard" img-dir)))))
        (ph/paragraph [:c-subheader] "Status and monthly trends.")
        [:spacer 2]))
   (ph/row (ph/cell [] (project-card/pdf-element-code-health period-renderable-trends img-dir)))))


(defn code-red [{:keys [code-red-url img-dir]}]
  (let [diagram (styling/load-resource img-dir "codered.png")
        text1 "The maximum time to implement an issue in Unhealthy code (red) is more than 9 times longer and leads to uncertainty during development."
        text2 "To a developer, uncertainty causes stress, over-time, and missed deadlines. For a product owner, high uncertainty makes it impossible to keep commitments to customers and stakeholders."]
    (ph/table
      [:c-card-content-top-table] [4 7]
      (ph/row
        (ph/cell [:cr-diagram-cell] (ph/image [:cr-diagram-image] diagram))
        (ph/cell
          [:cr-text-cell]
          (ph/paragraph [:cr-title-paragraph] "Impact of unhealthy code")
          [:spacer 1]
          (ph/paragraph [:cr-text-paragraph] text1)
          [:spacer 1]
          (ph/paragraph [:cr-text-paragraph] text2)
          [:spacer 1]
          (ph/table
            [] [1 5]
            (pdf-elements/link-row code-red-url "Learn more" img-dir)))))))