(ns codescene.features.pm-data.shortcut.shortcut-api
  "Contains methods for fetching data over the Shortcut Rest API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require [clj-http.client :as client]
            [clojure.string :as str]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]
            [taoensso.timbre :as log]))

(def shortcut-api-url "https://api.app.shortcut.com")
(def ^:const default-http-page-size 25)

(defn api-v3-url [sub-url]
  (str "/api/v3" sub-url))

(defn- api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [api-client method url request-params]
  (let [params (meta-merge
                {:url url
                 :content-type :json
                 :accept       :json
                 :as           :json-strict
                 :request-method method}
                request-params)
        final-params (-> api-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url shortcut-api-url))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch Shortcut data from " (:url final-params))
      (client/request final-params))))

(defn- api-request
  [api-client method url request-params]
  (:body (api-request* api-client method url request-params)))

(defn- get-data [api-client url query-params]
  (api-request api-client :get url
               {:query-params   query-params}))

(defn- get-paged-data [api-client url query-params]
  (let [page-size default-http-page-size]
    (loop [url url
           query-params (assoc query-params "page_size" (str page-size))
           total-count nil
           acc-data []]
      (api-client/check-interrupt)
      (log/infof "Fetching paged shortcut data from  %s%s..."
                 url (if total-count (format " %s (of %s)" (count acc-data) total-count) ""))
      (let [{:keys [next data total]} (get-data api-client url query-params)]
        (if next
          (recur next nil total (into acc-data data))
          (into acc-data data))))))

(defn fetch-projects [api-client]
  (let [sub-url  (api-v3-url "/projects")]
    (get-data api-client sub-url {})))

(defn fetch-member [api-client]
  (let [sub-url  (api-v3-url "/member")]
    (get-data api-client sub-url {})))

(defn search-stories
  [api-client search-options]
  (let [{:keys [since project]} search-options
        sub-url  (api-v3-url "/search/stories")
        conditions (filter some?
                           [(if since
                              (format "updated:%s..*" (dates/date->string since))
                              "is:story")
                            (when project (str "project:" project))])
        query-params {"query" (str/join " " conditions)
                       ;"page_size" "1"}
                      }]
    (get-paged-data api-client sub-url query-params)))

(defn shortcut-auth
  [api-token]
  (reify api-client/ApiClient
    (augment-request [_this req] (assoc-in req [:headers "Shortcut-Token"] api-token))
    (on-expired-token [_this] nil)))

(comment
  (def api-client (shortcut-auth (System/getenv "SHORTCUT_TOKEN")))
  (fetch-member api-client)
  (fetch-projects api-client)
  (clojure.pprint/pprint (search-stories api-client {:since (dates/string->date "2022-02-10")
                                                     :project 2}))
  )