(ns codescene.features.pm-data.github.github-provider
  "A PM Data Provider for fetching data from GitHub.
   Uses a cache for fetching data. 
   This means that a single instance of the Provider will only ever fetch data once,
   and a new instance (with a new cache) is needed for fetching updated data."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.github.github-fetcher :as f]
            [codescene.features.pm-data.github.github-cache :as gc]
            [codescene.features.pm-data.github.github-ticket-id-extractor :as x]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider]]
            [codescene.url.url-utils :as url]
            [taoensso.timbre :as log]))

(defn- with-href [host owner repo issues]
  (let [->href #(format "https://%s/%s/%s/issues/%s" host owner repo %)]
    (map #(assoc % :href (->href (:id %)))
         issues)))

(defn- with-repo-path [owner repo issues]
  (let [add-repo-path #(format "%s/%s#%s" owner repo %)]
    (map #(update % :id add-repo-path) issues)))

(defn fetch-project [provider-def repo-url since]
  (if-let [{:keys [host owner repo]} (url/github-repo-url->parts repo-url)]
    (let [{:keys [api-client]} provider-def
          issues (->> (gc/fetch-issues since api-client owner repo provider-def)
                      (with-href host owner repo)
                      (with-repo-path owner repo))]
      (log/infof "Import %d issues from GitHub repo %s" (count issues) repo-url)
      issues)
    []))

(defn- get-pm-data
  [provider-def {:keys [repo-urls since] :as _pm-data-context}]
  (let [issues (->> repo-urls
                    (mapcat #(fetch-project provider-def % since))
                    (into []))]
    {:issues issues
     :provider-def provider-def}))

(defn- fetch-labels [provider-def repo-urls]
  (->> repo-urls
       (keep #(url/github-repo-url->parts %))
       (f/fetch-labels (:api-client provider-def))))

(defn- get-configuration-data [provider-def {:keys [repo-urls] :as pm-data-context}]
  (log/info "Fetching configuration data from GitHub")
  {:work-types (fetch-labels provider-def repo-urls)
   :projects nil
   :transitions nil})

(defn- validate-settings
  [{:keys [api-client] :as _provider-def} {:keys [repo-urls] :as _pm-data-context}]
  (let [{:keys [owner]} (url/github-repo-url->parts (first repo-urls))]
    (f/fetch-user-type api-client owner)
    nil))

(defrecord GitHubProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this repo] (x/make-ticket-id-extractor provider-def repo pm-data-context))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def pm-data-context))
  (-validate-settings [_this] (validate-settings provider-def pm-data-context)))

(defmethod -create-pm-data-provider "github"
  [provider-def pm-data-context]
  (->GitHubProvider provider-def pm-data-context))

(comment
  (def repo-url "https://github.com/dotnet/aspnetcore.git")
  (def since (evolutionary-metrics.trends.dates/string->date "2024-09-01"))
  (def context {:repo-urls [repo-url]
                :since since})
  (def provider-def {:supported-work-types ["bug"]
                     :rename-work-types []
                     :defect-and-failure-labels []
                     :work-in-progress-transition-names ["In Progress"]
                     :work-done-transition-names ["Done"]
                     :api-client (System/getenv "GITHUB_TOKEN")})
  (get-configuration-data provider-def context)
  (validate-settings provider-def context)
  (get-pm-data provider-def context)
  )
