(ns codescene.features.code-coverage.parsers.lcov-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :branch-coverage
                                  :function-coverage])

(defn- end-of-record? [l]
  (str/starts-with? l "end_of_record"))

(defn- ->record [lines]
  (->> lines
       (mapv #(str/split % #":" 2))
       (group-by first)
       (m/map-vals (partial map second))))

(defn- parse-entries
  "Parses entries and returns two sets [covered all]"
  [parse-value-fn coll]
  (reduce (fn [[covered all] value]
            (let [{:keys [id covered-uncovered?]} (parse-value-fn value)]
              [(cond-> covered covered-uncovered? (conj id))
               (conj all id)]))
          [#{} #{}]
          coll))

(defn- parse-lines
  "Parses DA entries abd returns two sets [covered all]
      DA: The record identifier, indicating line data.
      <line number>: The specific line number within the source file.
      <execution count>: The number of times the line was executed during testing.
      [,<checksum>]: An optional MD5 checksum of the line."
  [da-coll]
  (let [->line-stat (fn [covered-uncovered-fn? value]
                      (let [[line hits] (take 2 (str/split value #","))]
                        {:id line
                         :covered-uncovered? (covered-uncovered-fn? hits)}))
        covered?  #(some-> % Integer/parseInt pos?)
        uncovered #(some-> % Integer/parseInt zero?)]
    {:lines-covered   (parse-entries (partial ->line-stat covered?)  da-coll)
     :lines-uncovered (parse-entries (partial ->line-stat uncovered) da-coll)}))

(defn- parse-branches
  "Parses BRDA entries and returns two sets [covered all]"
  [bdra-coll]
  (parse-entries (fn [value]
                   (let [[line block branch taken] (str/split value #"," 4)]
                     {:id [line block branch]
                      :covered-uncovered? (nil? (#{"-" "0"} taken))}))
                 bdra-coll))

(defn- parse-single-integer-value
  "Parses single integer value, the first element will be used if value coresponding to the key is a collection"
  [m key]
  (let [v (get m key)
        e (cond-> v
                  (coll? v) first)]
    (cond
      (or (nil? e) (number? e)) e
      (re-matches #"\d*" e) (Integer/parseInt e))))

(defn- detailed-line-coverage-depending-on
  [parse-options
   {:keys [lines-covered lines-uncovered] :as _line-coverage-stats}]
  (when (utils/needs-detailed-line-coverage? parse-options)
    (let [line-numbers-from first] ; the parsed coverage data is given as a tuple [matching-lines all-lines]
      {:details {:covered   (line-numbers-from lines-covered)
                 :uncovered (line-numbers-from lines-uncovered)}})))

(defn- record->coverage-entry 
  [parse-options
   r]
  ;; See also https://github.com/linux-test-project/lcov/issues/113
  (let [SF (-> (get r "SF") first (str/replace "\\" "/"))
        {:keys [lines-covered] :as line-coverage-stats} (parse-lines (get r "DA"))
        branch-data (parse-branches (get r "BRDA"))
        ;; the LCOV file contains for each file entry the covered number of functions "FNH" and the total number of functions "FNF"
        ;; https://github.com/linux-test-project/lcov/blob/master/man/geninfo.1#L1501
        function-data (utils/->entry (parse-single-integer-value r "FNH") (parse-single-integer-value r "FNF"))
        ->coverage (fn [[covered all]]
                     (when (seq all)
                       (utils/->entry (count covered) (count all))))]
    (m/assoc-some {:path SF
                   :name (-> SF (str/split #"/") last)}
                  :line-coverage (merge 
                                  (->coverage lines-covered)
                                  (detailed-line-coverage-depending-on parse-options line-coverage-stats))
                  :branch-coverage (->coverage branch-data)
                  :function-coverage function-data)))

(defn- records
  [lines]
  (when (seq lines)
    (let [record (->> (take-while (complement end-of-record?) lines) ->record)
          rest (->> (drop-while (complement end-of-record?) lines) (drop 1))]
      (lazy-seq
       (cons record
             (records rest))))))

(defn- read-coverage* [r parse-options]
  (->> (line-seq r)
       records
       (map (partial record->coverage-entry parse-options))))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "lcov")
    (-name [this] "LCov")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/lcov.info")
  (def parse-options {:cli-command "check"})
  (->> (read-coverage f parse-options))
  )
