// @flow
import $ from 'jquery';

/*
  This module encapsulates common logic for dealing with git remote repo paths validation.
  (see `validateRepoPathsOnServer` exported function).
  It's reused by new project wizard and project configuration page.

  The idea is to gather values of all input fields representing repo paths and validate
  data via AJAX call (POST /projects/validate-repo-paths.json).
  If any warning/error is found than it's shown using divs which are supposed to be right after
  the `.repo-path-input` elements (see `updateRepoPathAlert` function).
*/

function updateRepoPathAlert(repoPath: string, html: string, cssClass?: string) {
  const repoPathAlert = $('.repo-path-input')
    .filter(function() { return this.value === repoPath; })
    .next();

  repoPathAlert.html('');
  // notice that 'alert' class must also be removed/added - because otherwise there would be gaps in HTML
  repoPathAlert.removeClass('alert alert-warning alert-danger');

  repoPathAlert.html(html);
  if (cssClass) {
    repoPathAlert.addClass('alert ' + cssClass);
  }
}

function wrapInLink(path, fragment, content) {
  if (path) {
    return `<a href="${path}#${fragment}">${content}</a>`;
  }
  return content;
}

function showErrors(validationResult, erroneousPaths) {
  if (erroneousPaths) {
    erroneousPaths.forEach(
      (repoPath) => updateRepoPathAlert(
        repoPath,
        'No unique local name could be created for this repository. Make sure there are no duplicate repositories in your project.',
        'alert-danger')
    );
  }
}

function showWarnings(validationResult, erroneousPaths, projectConfigUrlPath) {
  if (validationResult.warnings) {
    Object.entries(validationResult.warnings).forEach(function ([repoName, repoPaths]) {
      Object.entries(repoPaths).forEach(function ([repoPath, qualifiedRepoName]) {
        // don't overwrite errors with warnings!
        if (!erroneousPaths || erroneousPaths.indexOf(repoPath) == -1) {
          updateRepoPathAlert(
            repoPath,
            `This repository will be cloned using <b><i>${qualifiedRepoName}</i></b> as a repo root directory.<br>
A simpler name <i>${repoName}</i> cannot be used due to conflicts with other repositories in this project.<br>
Make sure you use the proper repo name in your
${wrapInLink(projectConfigUrlPath, 'exclusions-and-filters', '<i>Exclude content</i> filters')},
${wrapInLink(projectConfigUrlPath, 'exclusions-and-filters', '<i>White List content</i> filters')},
${wrapInLink(projectConfigUrlPath, 'architecture', '<i>Architectural Components</i> definitions')},
and <a href="https://github.com/jenkinsci/codescene-plugin#configure-jenkins-for-codescene">Jenkins CI jobs</a> (if you use delta analysis).`,
            'alert-warning');
        }
      });
    } );
  }
}

function handleOks(validationResult) {
  if (validationResult.oks) {
    Object.keys(validationResult.oks).forEach(function (repoPath) {
      // clear any previous errors/warnings for this repo path
      updateRepoPathAlert(repoPath, '');
    });
  }
}

/**
 * Validates all inputs containing repository paths on the server calling
 * `/projects/validate-repo-paths.json` API.
 * Everything happens asynchronously, there's no useful value returned by this function!
 *
 * Deals with errors and warnings (always clearing them first).
 * Errors are supposed to be fixed by user (e.g. invalid git remote syntax or unresolvable conflict
 * in repos names).
 * Warnings can be automatically resolved on the server (using qualified repo names) but it's
 * useful to inform user so he can adjust the exclusion filters et al.
 *
 * Accepts optional `projectConfigUrlPath` - e.g. `/projects/95/configuration`.
 * If specified this will be used to include proper links to exclusion filters, arch. components, etc.
 */
export function validateRepoPathsOnServer(projectConfigUrlPath?: string) {
  const filledRepoFields = $('.repo-path-input').filter(function() { return this.value != ''; });
  const atLeastOneRepo = filledRepoFields.length >= 1;

  // clean up repo path alerts for empty repo paths -> there might be some leftovers (warnings/errors)
  updateRepoPathAlert('', '');

  if (atLeastOneRepo) {
    let repoPaths = filledRepoFields.map(function() { return this.value; }).get();
    $.post(`${window.baseUrl}projects/validate-repo-paths.json`,
      {'repo-paths': repoPaths},
      (validationResult) => {
        // intential: log server response for potential troubleshooting

        if (validationResult['invalid-git-remotes']) {
          Object.values(validationResult['invalid-git-remotes']).forEach(function (repoPath) {
            updateRepoPathAlert(
              repoPath,
              'Invalid git remote syntax. Please check your remote URL.',
              'alert-danger');
          });
        }

        const erroneousPaths = validationResult.errors
          ? Object.values(validationResult.errors).reduce((allPaths, paths) => allPaths.concat(paths), [])
          : null;

        showErrors(validationResult, erroneousPaths);

        showWarnings(validationResult, erroneousPaths, projectConfigUrlPath);

        handleOks(validationResult);
      })
      .fail((xhr) => alert('Cannot validate repo paths on server: ' + xhr.responseText));
  }
}

