// @flow
import $ from 'jquery';
import d3 from 'd3';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as alerts from 'codescene-ui/src/alerts';
import * as workTypeTrend from 'codescene-ui/src/analyses/work-type-trend/work-type-trend-chart';

var parseDate = d3.time.format('%Y-%m').parse;

function parseAndCountTotal(costUnit, row) {
  var total = 0;
  var parsed = {
    date: parseDate(row.date),
    total: 0
  };
  Object.keys(row)
    .filter(key => key !== 'date')
    .forEach(key => {
      const value = parseInt(row[key]);

      if (costUnit === 'minutes') {
        parsed[key] = (value / 60); // in hours
      } else if (costUnit === 'cycle-time-in-development') {
        parsed[key] = (value / 60); // in hours
      } else {
        parsed[key] = value;
      }
      
      total += value;
    });
  parsed.total = total;
  return parsed;
}

function presentableCostUnit(costUnit) {
  if (costUnit === 'minutes') {
    return 'Hours';
  }
  if (costUnit === 'cycle-time-in-development') {
    return 'Hours';
  }

  return costUnit;
}

function addCostConfidenceFrom(data, destinationElement) {
  if (destinationElement) {
    if (data.confidence) {
      const v = Math.round(parseFloat(data.confidence) * 100);
      $(destinationElement).append($('<strong>').append('Cost Confidence: ' + v + '%'));
    }
  }
}

type WorkTrendViewOptions = {
  workTypeTrendUrl : string,
  workItemType : string,
  costUnit : string,
  trendElement: string,
  trendElementProgress: string,
  costConfidenceElement: string,
  detailUrl: ?string,
  ticketIdUriTemplate: ?string
};

export function initWorkTrend(options : WorkTrendViewOptions) {
  const progressIndicator = alerts.progress($(options.trendElementProgress), 'Loading cost trend...');

  const workTypeTrendPromise = xhr.json(options.workTypeTrendUrl)
    .then(data => {
      const costs = data.costs.map(function (r) { return parseAndCountTotal(options.costUnit, r); });
      const health = data.health.map(function (r) { r.date = parseDate(r.date); return r; });

      workTypeTrend.displayTrendFrom(
        $(options.trendElement),
        costs,
        health,
        options.workItemType,
        presentableCostUnit(options.costUnit),
        options.detailUrl && function (period) {
          const separator = options.detailUrl.includes('?') ? '&' : '?';
          return `${options.detailUrl}${separator}period=${period}`;
        },
        options.ticketIdUriTemplate
      );

      addCostConfidenceFrom(data, options.costConfidenceElement);
    })
    .catch(err => {
      console.error(err.stack);
      $('#cost-download').remove();
      alerts.warning($(options.trendElement), 'No costs trend available -- enable Project Management Metrics in your configuration.');
    });

  Promise.all([workTypeTrendPromise])
    .then(() => {
      progressIndicator.remove();
    });
}

type ViewOptions = {
  workTypeTrendUrl : string,
  costUnit : string
};

export function init(options : ViewOptions) {
  const progressIndicator = alerts.progress($('.progress-indicator'), 'Loading cost trends...');

  const workTypeTrendPromise = xhr.csv(options.workTypeTrendUrl, function(r) { return parseAndCountTotal(options.costUnit, r); })
    .then(data => {
      workTypeTrend.displayTrendFrom($('#worktypetrendline'), data);
    })
    .catch(err => {
      console.error(err.stack);
      alerts.error($('#worktypetrendline'), 'Failed to load the Costs trend.');
    });

  Promise.all([workTypeTrendPromise])
    .then(() => {
      progressIndicator.remove();
    });
}
