// @flow
import d3 from 'd3';

type XhrFunction = (url : string) => Promise<any>;

function promisifyD3(methodName : string) : XhrFunction  {
  var method = d3[methodName];
  if (!method) {
    throw Error('No such method in d3: ' + methodName);
  }
  return function (url : string) {
    return new Promise(function (resolve, reject) {
      return method(url, function (err, data) {
        if (err) {
          reject(err);
        } else {
          resolve(data);
        }
      });
    });
  };
}

export var get : XhrFunction = promisifyD3('xhr');
export var json : XhrFunction = promisifyD3('json');

/**
 * Performs a GET request for the given URL, and parses the result as CSV. An
 * optional accessor function can be passed as the second argument, which is
 * mapped over each data row.
 *
 * Returns a promise of the data.
 */
export function csv() : Promise<any> {
  const [url, accessor] = arguments;

  return new Promise((resolve, reject) => {
    function callback (err, data) {
      if (err) {
        reject(err);
      } else {
        resolve(data);
      }
    }

    if (accessor) {
      d3.csv(url, accessor, callback);
    } else {
      d3.csv(url, callback);
    }
  });
}
