(ns mikera.cljutils.bytes
  "Utility functions fro working with byte arrays."
  (:refer-clojure :exclude [reverse])
  (:import [java.util Arrays])
  (:require [mikera.cljutils.hex :as hex])
  (:require [clojure.string :as str]))

(set! *warn-on-reflection* true)
(set! *unchecked-math* :warn-on-boxed)

(def BYTE-ARRAY-CLASS (Class/forName "[B"))

(defn reverse 
  (^bytes [^bytes bs]
    (let [n (alength bs)
          res (byte-array n)]
      (dotimes [i n]
        (aset res i (aget bs (- n (inc i)))))
      res)))

(defn join 
  "Concatenates two byte arrays"
  (^bytes [^bytes a ^bytes b]
    (let [al (int (alength a))
          bl (int (alength b))
          n (int (+ al bl))
          ^bytes res (byte-array n)]
      (System/arraycopy a (int 0) res (int 0) al)
      (System/arraycopy b (int 0) res (int al) bl)
      res)))

(defn slice
  "Slices a byte array with a given start and length"
  (^bytes [^bytes a ^long start]
    (slice a start (- (alength ^bytes a) start)))
  (^bytes [^bytes a ^long start ^long length]
    (let [al (int (alength ^bytes a))
          ^bytes res (byte-array length)]
      (System/arraycopy a (int start) res (int 0) length)
      res)))

(defn to-hex-string 
  "Converts a byte array to a string representation , with space as a default separator."
  ([^bytes bs]
    (to-hex-string bs " "))
  ([^bytes bs separator]
    (str/join separator (map #(hex/hex-string-from-byte %) bs))))

(defn unchecked-byte-array 
  "Like clojure.core/byte-array but performs unchecked casts on sequence values."
  (^bytes [size-or-seq] 
    (. clojure.lang.Numbers byte_array 
      (if (number? size-or-seq) 
        size-or-seq
        (map unchecked-byte size-or-seq ))))
  (^bytes [size init-val-or-seq] 
    (. clojure.lang.Numbers byte_array size 
      (if (sequential? init-val-or-seq) 
        (map unchecked-byte init-val-or-seq )
        init-val-or-seq))))

(defn bytes=
  "Compares two byte arrays for equality."
  ([^bytes a ^bytes b]
    (Arrays/equals a b)))