grammar Rust;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}

import RustSharedRulesLexer;

// Entry point when parsing a complete file (e.g. X-Ray)
//
translationunit
:
	expression* EOF
;

expression : function_declaration
           | test_module
           | containing_type
           | struct_protocol
           | top_level_block_statement
           | global_variable
           | anything;

anything : ~(LCURLYBRACE | RCURLYBRACE);

top_level_block_statement : LCURLYBRACE expression*? RCURLYBRACE;

// We want to identify embedded unit tests. In Rust, these tests are inside a test module. Syntax:
//
// #[cfg(test)]
//   mod tests {
//
// By identifying test code, we allow clients to use different code health configs for unit tests vs app code, even
// when all code is enclosed in the same file.
test_module: test_module_attribute containing_test_module_type;
test_module_attribute: POUND LSQUAREBRACKET 'cfg' LPAREN 'test' RPAREN RSQUAREBRACKET; // #[cfg(test)]
containing_test_module_type: KW_MOD containing_type_name ~(LCURLYBRACE | KW_FN)*? LCURLYBRACE expression* RCURLYBRACE;

containing_type: (KW_IMPL | KW_TRAIT | KW_MOD) generic_spec? containing_type_name ~(LCURLYBRACE | KW_FN)*? LCURLYBRACE expression* RCURLYBRACE;
containing_type_name: identifier;

// identify data members to calculate cohesion.
//
struct_protocol: KW_STRUCT identifier ~(LCURLYBRACE | KW_FN)*? LCURLYBRACE struct_expression* RCURLYBRACE;
struct_expression : member_declaration
                  | anything;
member_declaration: identifier COLON ~(RCURLYBRACE | LCURLYBRACE)*? COMMA;
global_variable: (KW_LET | KW_STATIC | KW_CONST) variable_modifiers? identifier
               | 'thread_local!' LPAREN KW_STATIC variable_modifiers identifier
               | 'lazy_static!' LPAREN KW_STATIC variable_modifiers identifier;
variable_modifiers: 'ref' | 'mut';

function_declaration: KW_FN identifier generic_spec? function_params function_return_spec? LCURLYBRACE function_body RCURLYBRACE;

identifier
   : LIFETIME_OR_LABEL
   | NON_KEYWORD_IDENTIFIER
   | RAW_IDENTIFIER
   | 'macro_rules'
   ;

function_return_spec: RARROW ~(LCURLYBRACE)+?;

// Note the option COMMA: we came across code that has a trailing comma separator after the
// last argument:
// fn transform_changes<'a>(
//       changes: &Vec<Change>,
//   ) -> HashMap<(LineNumber, Column), TransformedChange> {
function_params: LPAREN function_params_list? COMMA? RPAREN;

function_params_list: function_param
                    | function_params_list COMMA function_param;

function_param: selfParam
              | function_pointer_param
              | '&'? 'mut'? identifier COLON function_param_type
              ;

function_pointer_param: identifier COLON '&'? 'dyn'? function_pointer_param_type;

function_pointer_param_type: KW_FN LPAREN function_pointer_params_list? RPAREN  // f: &dyn fn(i32)
                             RARROW ~(COMMA | RPAREN | LCURLYBRACE)+?;          // -> i32
function_pointer_params_list: function_param_type
                            | function_pointer_params_list COMMA function_param_type;

function_param_type: ~(COMMA | RPAREN)+?;
selfParam
   : outerAttribute* (shorthandSelf | typedSelf);
outerAttribute
   : '#' '[' ~(']')*? ']';
shorthandSelf
   : ('&' lifetime?)? 'mut'? 'self';
lifetime
   : LIFETIME_OR_LABEL
   | '\'static'
   | '\'_';
typedSelf
   : 'mut'? 'self' ':' function_param_type;

function_body: function_body_expression*;

function_body_expression: function_body_block
                        | anything;
function_body_block: LCURLYBRACE function_body_expression*? RCURLYBRACE;

generic_spec: LT generic_arg_list GT;
generic_arg: 'const'? identifier (COLON generic_type)? generic_default_value?;
generic_default_value: '=' .;  // Mul<Output = T>
generic_type: (identifier generic_spec generic_trait) // fn square<T: Mul<Output = T>> (x: T) -> T {
            | ~('<' | '>')*?; // plain non-generic type
generic_trait: ~(COMMA | '<' | '>')*;
generic_arg_list: generic_arg
                 | generic_arg_list COMMA generic_arg;

// Entry point for cyclomatic complexity.
//
cyclomatic_complexity
:
	cc_expression* EOF
;

cc_expression: cc_introducing_construct
             | cc_anything;

cc_introducing_construct: conditional
                        | loop
                        | cc_operators;

conditional: KW_IF | KW_MATCH;

loop: KW_LOOP | KW_WHILE | KW_FOR;

cc_operators: ANDAND | OROR;

cc_anything: .;

// Entry point for nesting logic.
//
nested_complexity
:
	nested_complexity_expression* EOF
;

nested_complexity_expression: nesting_expression
                            | cc_anything;

nesting_expression: conditional_clause
                  | loop_clause
                  | iteration_clause;

conditional_clause: conditional some_condition multi_line_conditional_expression;

iteration_clause: (KW_WHILE | KW_FOR) some_condition multi_line_conditional_expression;

loop_clause: KW_LOOP multi_line_conditional_expression;

some_condition: (conditional_operator
              |  ~(LCURLYBRACE | SEMI | KW_LOOP | KW_WHILE | KW_FOR))+;
conditional_operator: ANDAND | OROR;

multi_line_conditional_expression: LCURLYBRACE nested_complexity_expression*? RCURLYBRACE;

// Entry point for test code smells.
//
test_code_single_function_scope
:
	test_code_expression* EOF
;

test_code_expression : assertion_blocks
                     | .;

assertion_blocks: assertion_statement+;
assertion_statement: (KW_ASSERT | KW_ASSERT_EQ | KW_ASSERT_NE) assert_condition SEMI?;

assert_condition: LPAREN assert_parts*? RPAREN;

assert_parts: ~(LPAREN | RPAREN | KW_ASSERT | KW_ASSERT_EQ | KW_ASSERT_NE)
            | LPAREN assert_parts* RPAREN;
