grammar ObjectiveCNestedComplexity;

// This grammar detects islands of nested complexity.
// Note that we have duplicated a fair chunk of the C++ parser, but the two aren't easily shared.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_clause
          | switch_clause
          | for_loop
          | for_each_loop
          | while_loop
          | do_while_loop;

block_expression: OPENING_BRACE expression*? CLOSING_BRACE;

anything: ~(OPENING_BRACE | CLOSING_BRACE);

if_clause: IF some_condition (multi_line_conditional_expression | plain_line);
else_clause: ELSE (multi_line_conditional_expression | plain_line);

switch_clause: SWITCH some_condition multi_line_conditional_expression;

multi_line_conditional_expression: OPENING_BRACE expression*? CLOSING_BRACE;
plain_line: ~OPENING_BRACE ~(SEMICOLON)*? SEMICOLON;

for_loop: FOR LEFT_PAREN for_loop_part SEMICOLON for_loop_part SEMICOLON for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_loop_part: ~(SEMICOLON)*?; // each part is optional in C
for_loop_condition: conditions*?;

for_each_loop: FOR LEFT_PAREN for_each_loop_part IN for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_each_loop_part:  ~(IN | SEMICOLON)+?;

while_loop: WHILE some_condition (multi_line_conditional_expression | plain_line);
do_while_loop: DO multi_line_conditional_expression WHILE some_condition;

some_condition: LEFT_PAREN conditions+ RIGHT_PAREN;
conditions : conditional_operator
           | ~(LEFT_PAREN | RIGHT_PAREN | SEMICOLON | OPENING_BRACE | IF | ELSE | FOR | WHILE | DO | SWITCH)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: AND | OR;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

IF: 'if';
ELSE: 'else';
FOR: 'for';
IN: 'in';
WHILE: 'while';
DO: 'do';
SWITCH: 'switch';
AND: '&&';
OR: '||';

ID : [a-zA-Z_][a-zA-Z0-9_]*;

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';
COLON: ':';
SEMICOLON: ';';

Whitespace : [ \t]+ -> skip;

PRE_PROC_STUFF: '#'  ~[\r\n]* -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence