(ns codescene.features.util.string
  "Handy utility functions for dealing with strings.
  Most of the time you should probably favor using `clojure.string` directly.")

(defn truncate
  "Truncate a string with suffix (ellipsis by default) only if it is
   longer than specified length."

  ([^String string length suffix skip]
   {:pre [(>= length 0) (>= skip 0)]}
   (when string
     (let [suffix-len (count suffix)
           skip-string (subs string 0 skip)
           remaining-string (subs string skip)
           string-len (.length remaining-string)]

       (if (<= string-len length)
         string
         (str skip-string (subs remaining-string 0 (- length suffix-len)) suffix)))))

  ([^String string length suffix]
   (truncate string length suffix 0))

  ([^String string length]
   (truncate string length "...")))

(defn hide
  "Truncate a 'string' to be max the given 'length' including the ellipsis,
   but hide parts of it even if it's shorter than 'length'."

  ([^String string length skip]
   {:pre [(>= length 0) (>= skip 0)]}
   (when string
     (let [skip-string (subs string 0 skip)
           remaining-string (subs string skip)
           string-len (.length remaining-string)]
       (if (<= string-len length)
         (str skip-string (subs remaining-string 0 (- string-len (/ string-len 2))) "...")
         (truncate string length "..." skip)))))

  ([^String string length]
   (hide string length 0)))

(defn safe-match-some?
  "Returns true if string matches regex at any position. nil or empty regex is considered to never
   match, nil string is considered to never match, even if regex is nil. Invalid regex never matches."
  [regex-str s]
  (try
    (boolean (and s (seq regex-str) (re-find (re-pattern regex-str) s)))
    (catch Exception _ false)))

(defn declob
  "Turn a clob into a String"
  [clob]
  (when clob
    (if (string? clob)
      clob
      (slurp (.getCharacterStream clob)))))