(ns codescene.features.tools.licensing
  "This ns contains fns for handling signed JWT tokens containing license claims
   that are passed from cloud/onprem to CLIs (and is used by both sides)"
  (:require [buddy.sign.jwt :as buddy-jwt]
            [codescene.features.util.jwt :as jwt]
            [clj-time.core :as tc]
            [clj-time.coerce :as tcc]
            [integrant.core :as ig]
            [medley.core :as m]))

(defprotocol LicenseProvider
  (-cli-license [this user]))

(defn claims [duration leeway issuer aud]
  (m/assoc-some {:iss issuer
                 :iat (tcc/to-epoch (tc/now))}
    :exp (some->> duration (tc/plus (tc/now)) tcc/to-epoch)
    :aud aud
    ;; leeway is how many seconds after expiry the validation still works
    :leeway (when leeway (tc/in-seconds leeway))))

(defn jwt [claims keyring] (jwt/make-jwt claims nil keyring))

(defn expired? [claims] (<= (:exp claims) (tcc/to-epoch (tc/now))))

(defn verified-claims [license license-keyring]
  (let [typ (-> (jwt/extract-jwt license) :header :typ)]
    (if typ
      (jwt/unpack-jwt license license-keyring)
      ;; older onprem versions emit no typ
      (buddy-jwt/unsign license
                        (some #(when (= "csent" (:kid %)) (:public-key %)) license-keyring)
                        {:alg :eddsa}))))

(defrecord JWTLicenseProvider [f]
  LicenseProvider
  (-cli-license [this user] (f user "codescene-cli")))

(defn jwt-license-provider
  [issuer keyring duration-fn user-claims-fn]
  (let [keyring (jwt/init-keyring keyring)]
    (->JWTLicenseProvider
     (fn [user product]
       (jwt (merge (claims (duration-fn user) nil issuer product) 
                   (user-claims-fn user))
            keyring)))))

(defn cli-license [provider user] (-cli-license provider user))

(defmethod ig/init-key ::no-licensing [_ _v])

(derive ::no-licensing :codescene.features.tools.components/licensing)