(ns codescene.features.scheduling.persistent
  "Persistent scheduler backed by database tables (look for 'QRTZ_')."
  (:require
    [clojure.spec.alpha :as s]
    [taoensso.timbre :as log]
    [twarc.core :as twarc])
  (:import (org.quartz Scheduler JobKey)))

;; will be initialized and started when init is called
;; Beware that when you eval this file during development, you'll reset this var back to nil!
;; => you need to call `destroy` and `init` again
(defonce ^:dynamic *persistent-sched* nil)

(defn scheduler
  "Returns the persistent scheduler.
  Prefer this to directly accessing `*persistent-sched*`."
  []
  (assert *persistent-sched* "Scheduler has not been initialized; run init")
  *persistent-sched*)

(s/fdef job-data :args (s/cat :job-key #(instance? JobKey %)))
(defn job-data [job-key]
  (when-let [{:strs [arguments state]} (some->>
                                         job-key
                                         (.getJobDetail ^Scheduler (:twarc/quartz (scheduler)))
                                         (.getJobDataMap))]
    {:args arguments
     :state state
     :job-key job-key}))

(def props {:threadPool.class                           "org.quartz.simpl.SimpleThreadPool"
            :threadPool.threadCount                     2
            :plugin.triggHistory.class                  "org.quartz.plugins.history.LoggingTriggerHistoryPlugin"
            :plugin.triggHistory.triggerFiredMessage    "Trigger [{1}.{0}] fired job [{6}.{5}] scheduled at: {2, date, yyyy-MM-dd HH:mm:ss.SSS}, next scheduled at: {3, date, yyyy-MM-dd HH:mm:ss.SSS}"
            :plugin.triggHistory.triggerCompleteMessage "Trigger [{1}.{0}] completed firing job [{6}.{5}] with resulting trigger instruction code: {9}. Next scheduled at: {3, date, yyyy-MM-dd HH:mm:ss.SSS}"
            :plugin.triggHistory.triggerMisfiredMessage "Trigger [{1}.{0}] misfired job [{6}.{5}]. Should have fired at: {3, date, yyyy-MM-dd HH:mm:ss.SSS}"
            :plugin.jobHistory.class                    "org.quartz.plugins.history.LoggingJobHistoryPlugin"
            :plugin.jobHistory.jobToBeFiredMessage      "Job [{1}.{0}] to be fired by trigger [{4}.{3}], re-fire: {7}"
            :plugin.jobHistory.jobSuccessMessage        "Job [{1}.{0}] execution complete and reports: {8}"
            :plugin.jobHistory.jobFailedMessage         "Job [{1}.{0}] execution failed with exception: {8}"
            :plugin.jobHistory.jobWasVetoedMessage      "Job [{1}.{0}] was vetoed. It was to be fired by trigger [{4}.{3}] at: {2, date, yyyy-MM-dd HH:mm:ss.SSS}"
            :scheduler.skipUpdateCheck                  true
            :scheduler.instanceId                       "AUTO"
            :scheduler.instanceName                     "CodeSceneClusteredScheduler"})

(defn persistent-props [driver-class db-url db-user db-password]
  {:jobStore.class "org.quartz.impl.jdbcjobstore.JobStoreTX"
   :jobStore.driverDelegateClass "org.quartz.impl.jdbcjobstore.StdJDBCDelegate"
   :jobStore.tablePrefix "QRTZ_"
   :jobStore.dataSource "db"
   :jobStore.isClustered true
   :dataSource.db.maxConnections 2
   :dataSource.db.driver driver-class
   :dataSource.db.URL db-url
   :dataSource.db.user db-user
   :dataSource.db.password db-password
   :dataSource.db.maxIdleTime 50})

(defn execution-context
  "Returns JobExecutionContext of the job's scheduler parameter"
  [context-scheduler]
  (:twarc/execution-context context-scheduler))

(defn job-keys-by-group
  [scheduler group-name]
  (let [matcher (twarc/matcher {:group [:equals group-name]})
        quartz-scheduler (:twarc/quartz scheduler)]
    (.getJobKeys quartz-scheduler matcher)))

(defn jobs-data-by-group
  [scheduler group-name]
  (let [job-keys (job-keys-by-group scheduler group-name)]
    (mapv #(job-data %) job-keys)))

(defn delete-job-group
  "Deletes all jobs with the specified job group"
  [scheduler group-name]
  (.deleteJobs (:twarc/quartz scheduler) (vec (job-keys-by-group scheduler group-name))))

(defn ^JobKey current-job-key
  "Returns JobKey for job currently associated with the scheduler."
  [context-scheduler]
  (.. (execution-context context-scheduler) (getJobDetail) (getKey)))

(defn delete-current-job
  "Deletes the job currently associated with the scheduler."
  [context-scheduler]
  (.deleteJob (:twarc/quartz context-scheduler) (current-job-key context-scheduler)))

(defn make-scheduler [jobstore-props]
  (-> (twarc/make-scheduler (merge props jobstore-props) {:name "main-sched"})
      (twarc/start)))

(defn init
  "Starts the scheduler."
  [jobstore-props]
  (log/info "Starting Quartz persistent scheduler")
  ;; see http://www.quartz-scheduler.org/documentation/quartz-2.0.2/configuration/ConfigDataSources.html
  (alter-var-root #'*persistent-sched*
                  (constantly (make-scheduler jobstore-props))))

(defn destroy []
  (log/info "Stopping Quartz persistent scheduler")
  (when *persistent-sched*
    (twarc/stop *persistent-sched*)
    (alter-var-root #'*persistent-sched* (constantly nil))))

(defn find-jobs
  "This is a helper function for operations, not routine use within the app.
  It can be used by people examining and fixing scheduled jobs manually,
  e.g. when deleting scheduled reports associated with bouncing emails."
  [job-group-prefix]
  (let [job-matcher (twarc/matcher {:group [:starts-with job-group-prefix]})
        scheduled-reports-jobs (.getJobKeys (:twarc/quartz (scheduler)) job-matcher)]
    (mapv job-data scheduled-reports-jobs)))

;; get job data for all scheduled reports
(comment
  (def all-scheduled-reports
    {:account-reports (find-jobs "account-report-")
     :project-reports (find-jobs "project-report-")})
  .)

(defmacro with-test-scheduler [& body]
  `(binding [*persistent-sched* (make-scheduler {:jobStore.class "org.quartz.simpl.RAMJobStore"})]
     (try
       ~@body
       (finally (twarc/stop (scheduler))))))

(defn scheduler-fixture [f]
  (with-test-scheduler (f)))