(ns codescene.features.repository-provider.gitlab.api
  "For GitLab we'll be using both GraphQL and REST API. Publicly GitLab is saying they want to get
  people on GraphQL as the primary API, but there's several problems with it:
  - most connections don't have any filtering options at all
  - some options just aren't there, like selecting Group by ID (????)
  - webhooks are completely absent

  REST version has other problems:
  - you can filter by access level but it doesn't return the actual access level
  - it returns much much more data even when we need just 3 fields
  - selecting all branches of all repos in project is going to be really slow

  Of course the ID scheme between the two doesn't match."
  (:require [clj-http.client :as http-client]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.spec.alpha :as s]
            [codescene.specs :as base-specs]
            [codescene.features.client.api :as api-client]
            [codescene.features.client.graphql :as gql]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.repository-provider.specs :as specs]
            [codescene.features.util.maps :refer [map-of ->kebab] :as maps]
            [codescene.features.util.url :refer [make-url]]
            [evolutionary-metrics.trends.dates :as dates]
            [medley.core :refer [assoc-some index-by filter-keys]]
            [meta-merge.core :refer [meta-merge]]
            [org.clojars.roklenarcic.paginator :as page]
            [ring.util.codec :as codec]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(def ^:private pagelen 100)

(def max-parallelism
  "Maximum number of concurrent requests to GitLab when processing one action e.g.
  loading all branches from multiple repositories."
  5)

(s/def ::gid #(re-find #"^gid://gitlab/[^/]++/.*$" %))

(s/def ::str-int (s/or :str ::base-specs/non-empty-string :int integer?))

(defn graphql-assert-success!
  "Parse GraphQL response and if there were errors, throw a Bad Request exception."
  [resp]
  (when-let [errors (-> resp :body :errors)]
    (throw
      (api-client/service-exception
        :gitlab
        (str/join "\n" (map :message errors))
        {:type :http-error
         :status 400
         :errors errors}))))

(defn- process-exception
  [exception {:keys [url] :as params}]
  (let [{:keys [body status]} (ex-data exception)
        parsed-body (api-client/json-safe-parse body)
        message (or (:error_description parsed-body)
                    (:error parsed-body) 
                    (ex-message exception))
        error-type (:error parsed-body)
        token-expired? (or (= error-type "invalid_token")
                           (some #(= "Invalid token" (:message %)) (:errors parsed-body)))
        data (map-of message url)]
    (log/debugf "GitLab call returned %s %s" status body)
    (log/info exception "GitLab call exception " (api-client/printable-req params))
    (case status
      401 (api-client/ex-unauthorized :gitlab exception (assoc data :token-expired? token-expired?))
      403 (api-client/ex-forbidden :gitlab exception data)
      404 (api-client/ex-forbidden :gitlab exception data)
      (api-client/ex-http-error :gitlab exception (or status 0) data))))

(defn split-path
  "Separates full path to prefix and last path segment.

  Important: when selecting a project you get 'path', 'fullPath' and you
  can also get namespace.fullPath. The easiest would be to use
  project's 'path' and repository repo-slug and namespace.fullPath as owner-login.

  This doesn't always work, because I've seen public projects in private groups return
  the correct fullPath, but the namespace is not accessible in graphql and you get nil there.

  So we should split the fullPath ourselves rather than hoping for namespace being available."
  [full-path]
  (some->> full-path (re-find #"(.+?)/([^/]+)$") rest))

(defn apply-client [authed-client params]
  (-> authed-client
      (api-client/augment-request params)
      (api-client/fix-api-url "https://gitlab.com/api/v4")))

(defn request-url [authed-client url] (:url (apply-client authed-client {:url url})))

(s/fdef api-request
        :args (s/cat :auth-token ::specs/auth-token
                     :method keyword?
                     :url ::specs/url
                     :params map?))
(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :as :json
                  :content-type :json
                  :request-method method}
                 request-params)
        graphql? (= url "/graphql")
        final-params (-> (apply-client authed-client params)
                         (cond-> graphql? (update :url #(str/replace-first % #"^(https?://.+/api)/v4/graphql$" "$1/graphql"))))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    (try
      (assoc (http-client/request final-params) :url (:url final-params))
      (catch Exception e
        (-> e
            (process-exception final-params)
            (api-client/retry-token authed-client)
            (api-request* method url request-params))))))

(defn api-request [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(s/fdef graphql-request
        :args (s/cat :auth-token ::specs/auth-token
                     :query string?))
(defn graphql-request
  "Core function for GraphQl requests.  Requires ApiClient token.

   Uses augment-request to add some sort of credentials to the request."
  [authed-client query]
  (log/debugf "Running GitLab GQL query %s" query)
  (api-request* authed-client :post "/graphql" {:socket-timeout 60000
                                                :form-params {:query query}}))

(s/fdef graphql-query
        :args (s/cat :auth-token ::specs/auth-token
                     :query string?))
(defn graphql-query
  "Performs a graphql query.

   The GitLab GraphQL simply doesn't return data under currentUser key, for instance,
   when token is not valid.

   If request doesn't return currentUser.id then on-expired-token will be called on auth-token.

   If that returns a ApiClient, the request will be retried."
  [authed-client query]
  (let [resp (graphql-request authed-client (format "query { authCheck: currentUser { id }\n%s\n}" query))]
    (graphql-assert-success! resp)
    (if (or (nil? authed-client) (-> resp :body :data :authCheck some?))
      resp
      ;; TODO ROK TEMPORARY... see if this error mechanism is still used by GitLab, remove auth check query otherwise
      (do (log/warn "Auth Check Indicates Unauthorized Response")
          (-> (api-client/ex-unauthorized
                :gitlab
                {:resp resp}
                {:message "GraphQL response seems unauthorized" :token-expired? true})
              (api-client/retry-token authed-client)
              (graphql-query query))))))

(defn api-page-request
  "A helper function for a request that requests a particular page. It is based on a paging state
  and returns an items+cursor map."
  [authed-client {:keys [cursor add-page]} url request-params]
  (api-client/check-interrupt)
  (let [resp (if cursor
               (api-request* authed-client :get cursor (dissoc request-params :query-params))
               (api-request* authed-client :get url request-params))]
    (add-page (get-in resp [:body])
              (get-in resp [:links :next :href]))))

(defn graphql-page-request
  "A helper function for a graphql request that requests a page via the standard GraphQL paging.
  Returns an items+cursor map. Expects query to have ::paging marker."
  [authed-client {:keys [cursor add-page]} query path-to-connection]
  (api-client/check-interrupt)
  (let [paging-subs #(cond-> (dissoc %2 ::paging)
                       (::paging %2) (assoc-some :first pagelen :after (when cursor (str cursor))))
        final-query (gql/generate-query (gql/update-arguments paging-subs query))
        resp (get-in (graphql-query authed-client final-query) (into [:body :data] path-to-connection))]
    (add-page (:nodes resp [])
              (when-let [{:keys [hasNextPage endCursor]} (:pageInfo resp)]
                (when hasNextPage endCursor)))))

(defn get-all-pages [authed-client url params]
  (-> #(api-page-request authed-client % url params)
      page/paginate-one!
      page/unwrap))

(def maintainer-access-level
  "https://docs.gitlab.com/ee/development/permissions.html#members"
  40)

(defn access-level->org-role
  [access-level]
  (if (>= access-level maintainer-access-level) :admin :member))

(s/fdef gid->id :args (s/cat :gid ::gid))
(defn gid->id
  "Transforms GraphQL global ID to a REST ID"
  [gid]
  (second (re-find #"^gid://gitlab/[^/]++/(.*)$" gid)))

(defn repo-info->project-id
  "Returns project ID as expected in REST API URLs"
  [{:keys [external-id repo-slug owner-login]}]
  (if external-id
    (gid->id external-id)
    (codec/url-encode (str owner-login "/" repo-slug))))

(s/fdef project-id->gid :args (s/cat :id ::str-int))
(defn project-id->gid [project-id] (format "gid://gitlab/Project/%s" project-id))

(s/fdef group-id->gid :args (s/cat :id ::str-int))
(defn group-id->gid
  [group-id]
  (format "gid://gitlab/Group/%s" group-id))

(s/fdef get-group :args (s/cat :auth-token ::specs/auth-token
                               :gid ::base-specs/non-empty-string))
(defn get-group
  "Get group information of the group by ID."
  [authed-client id]
  (api-request
    authed-client
    :get
    (format "groups/%s" id)
    {:query-params {"with_projects" "false"}}))

(s/fdef get-group-members :args (s/cat :auth-token ::specs/auth-token
                                       :id ::base-specs/non-empty-string))
(defn get-group-members
  "Get group members of the group by ID."
  [authed-client id]
  (get-all-pages
    authed-client
    (format "groups/%s/members/all" id)
    {:query-params {"with_projects" "false"
                    "per_page" pagelen}}))

(defn- response-seq [resp cnt]
  (eduction (map #(keyword (str "part_" %)))
            (map (-> resp :body :data))
            (range cnt)))

(defn get-groups-by-names 
  "Gets basic group info selected by group names."
  [authed-client names]
  (if (> (count names) 40)
    (lazy-cat (get-groups-by-names authed-client (take 40 names))
              (get-groups-by-names authed-client (drop 40 names)))
    (let [query (map-indexed (fn [idx n]
                               `{(~(gql/as (str "part_" idx) :group) {:fullPath ~n})
                                 [:id :name :parent [:id] :fullPath]}) names)]
      (->> (response-seq (graphql-query authed-client (gql/generate-query (apply merge query)))
                         (count names))
           (keep (fn [g] (and g (update g :id gid->id))))))))

(s/fdef user-group-access :args (s/cat :auth-token ::specs/auth-token 
                                           :id ::base-specs/non-empty-string 
                                           :user-id ::str-int))
(defn user-group-access
  "Return data for user's access level in a group, nil otherwise."
  [authed-client id user-id]
  (let [ret (api-request authed-client :get (format "groups/%s/members/all" id)
                         {:throw-exceptions false
                          :query-params {"with_projects" "false"
                                         "user_ids[]" user-id}})]
    (:access_level (first ret))))

(s/fdef get-group-projects :args (s/cat :auth-token ::specs/auth-token
                                        :id ::base-specs/non-empty-string
                                        :private? boolean?))
(defn get-group-projects
  "Get group projects of the group by ID."
  [authed-client id private?]
  (get-all-pages
    authed-client
    (format "groups/%s/projects" id)
    {:query-params (assoc-some
                     {"order_by" "name"
                      "sort" "asc"
                      "include_subgroups" "true"
                      "min_access_level" 10
                      "per_page" pagelen}
                     "visibility" (when-not private? "public"))
     ;; For large gitlab accounts, it make take significantly longer than 10 seconds to load all the repositories
     :socket-timeout 60000}))

(s/fdef get-groups :args (s/cat :auth-token ::specs/auth-token
                                :top-level-only? boolean?))
(defn get-groups
  "Get group information for the current user"
  [authed-client top-level-only?]
  (let [toplevel? #(-> % :group :parent nil?)
        xf (fn [data]
             ;; only return top level groups
             (let [role (access-level->org-role
                          (-> data :accessLevel :integerValue))]
               (when (or (toplevel? data) (not top-level-only?))
                 (-> data
                     (assoc :role role)
                     (update-in [:group :id] gid->id)))))]
    (-> (fn [paging-state]
          (let [query {:currentUser
                       {`(groupMemberships {::paging true})
                        {:pageInfo [:hasNextPage :endCursor]
                         :nodes {:accessLevel [:integerValue]
                                 :group [:id :name :parent [:id] :fullPath]}}}}
                paging-state (api-client/with-item-xf paging-state (keep xf))]
            (graphql-page-request authed-client paging-state query [:currentUser :groupMemberships])))
        page/paginate-one!
        page/unwrap)))

(s/fdef shared-groups :args (s/cat :auth-token ::specs/auth-token 
                                  :group-id string?))
(defn shared-groups [authed-client group-id]
  (->> (get-all-pages authed-client
                      (api-client/to-url "/groups" group-id "groups/shared")
                      {:query-params {"per_page" pagelen}})
       (mapv (fn [{:keys [parent_id name id full_path]}]
               {:group {:id (str id) :name name :parent parent_id :fullPath full_path}}))))

(defn get-toplevel-groups-by-ids
  [authed-client ids]
  (let [xf (fn [data]
             ;; only return top level groups
             (let [role (access-level->org-role
                          (-> data :maxAccessLevel :integerValue))]
               (-> data
                   (assoc :role role)
                   (update :id gid->id))))]
    (-> (fn [paging-state]
          (let [query {`(groups {::paging true :ids ~(mapv group-id->gid ids) :topLevelOnly true})
                       {:pageInfo [:hasNextPage :endCursor]
                        :nodes [:maxAccessLevel [:integerValue :humanAccess :stringValue]
                                :id :name :parent [:id] :fullPath]}}
                paging-state (api-client/with-item-xf paging-state (keep xf))]
            (graphql-page-request authed-client paging-state query [:groups])))
        page/paginate-one!
        page/unwrap)))

(s/fdef get-user-projects :args (s/cat :auth-token ::specs/auth-token
                                       :private-repos? boolean?))
(defn get-user-projects
  "Get project information for the current user"
  [authed-client private-repos?]
  (let [accessible? #(or private-repos? (= "public" (:visibility %)))]
    (->> (get-all-pages
           authed-client
           "/projects"
           {:query-params (assoc-some
                            {"order_by" "name"
                             "sort" "asc"
                             "membership" "true"
                             "min_access_level" 10
                             "per_page" pagelen}
                            "visibility" (when-not private-repos? "public"))
            ;; For large gitlab accounts, it make take significantly longer than 10 seconds to load all the repositories
            :socket-timeout 60000})
         (filterv accessible?))))

(defn project-ids-filter [repo-infos] {:ids (mapv :external-id repo-infos)})
(defn project-names-filter [repo-infos] {:fullPaths (mapv (fn [{:keys [owner-login repo-slug]}]
                                                      (str owner-login "/" repo-slug))
                                                    repo-infos)})

(defn term-size [term] (count (mapcat term [:ids :fullPaths])))

(s/fdef project-filter-terms
        :args (s/cat :repo-infos (s/coll-of ::specs/repo-info)))
(defn project-filter-terms [repo-infos]
  (let [{by-id true by-name false} (group-by #(some? (:external-id %)) repo-infos)]
    (concat (map project-ids-filter (partition-all pagelen by-id))
            (map project-names-filter (partition-all 50 by-name)))))

(s/fdef get-filtered-projects
        :args (s/cat :auth-token ::specs/auth-token
                     :filter-terms (s/coll-of map?)))
(defn get-filtered-projects
  "Each term is a list of IDs or paths of projects to match."
  [authed-client [term :as filter-terms]]
  (if (= (count filter-terms) 1)
    (let [q (gql/generate-query
              `{(projects ~(merge {:first (term-size term)} term))
                {:nodes [:id :path :name ...on Project [:namespace [:id]] :repository [:rootRef] :visibility :httpUrlToRepo :fullPath]}})]
      (get-in (graphql-query authed-client q) [:body :data :projects :nodes]))
    (->> filter-terms
         (map (page/async-fn #(get-filtered-projects authed-client [%]) max-parallelism))
         (mapcat page/unwrap-async))))

(defn get-merge-request-commits
  [authed-client repo-info pr-id]
  (->> (get-all-pages
         authed-client
         (format "projects/%s/merge_requests/%s/commits"
                 (repo-info->project-id repo-info)
                 pr-id)
         {:query-params {"per_page" pagelen}})
       (mapv :id)
       rseq
       vec))

(def repo-branches-per-page
  "Determined by testing.... it affects Query Complexity,
  setting it higher throws errors"
  40)

(defn get-branches-page
  "Fetches one page for multiple repositories and returns new paging states."
  [authed-client max-branches paging-states]
  (let [cursor (-> paging-states first :cursor (or 0))
        _ (log/debugf "Requesting branches for repos %s, offset %s"
                      (mapv :id paging-states)
                      cursor)
        resp (graphql-query authed-client
                            (gql/generate-query
                              `{(projects {:ids ~(map :id paging-states) :first ~(count paging-states)})
                                  {:nodes [:id
                                           :repository [:rootRef
                                                        (branchNames {:limit ~pagelen
                                                                      :offset ~cursor
                                                                      :searchPattern "*"})]]}}))
        nodes (index-by :id (get-in resp [:body :data :projects :nodes]))]
    (->> paging-states
         (mapv #(api-client/limit-items % max-branches))
         (mapv (fn [{:keys [add-page id]}]
                 (if-let [{:keys [branchNames rootRef]} (:repository (nodes id))]
                   (add-page branchNames
                             (when (= (count branchNames) pagelen)
                               (+ pagelen cursor))
                             {:default-branch rootRef})
                   (add-page nil nil providers/no-branch-results)))))))

(s/fdef get-branches
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-infos (s/coll-of ::specs/repo-info)
                     :max-branches pos-int?))
(defn get-branches
  "Returns a map of external ID to default branch + refs"
  [authed-client repo-infos max-branches]
  (log/infof "Listing branches for %d repos." (count repo-infos))
  (let [branch-repos-max-count 35 ; limit query complexity
        get-page (page/async-fn (partial get-branches-page authed-client max-branches) max-parallelism)]
    (->> (map :external-id repo-infos)
         (page/paginate! get-page {:batcher (page/grouped-batcher :cursor branch-repos-max-count)})
         (mapv providers/->branch-results))))

(defn repo-check-rest-api-adapter
  "For a specific client https://app.hubspot.com/contacts/7007026/record/0-5/20318482714/ the GraphQL
  query returns empty array for user's projects, no idea why. Gitlab is crap. So we will fall back and select all projects,
  sort out availability from there. The detection mechanism for this scenario isn't perfect, we just check if the
  GraphQL query returns a project, which can also happen if the user legitimately doesn't have any access."
  [authed-client repo-infos]
  (log/infof "Using alternative repo availability check for a project. %s" (into #{} (map :owner-login repo-infos)))
  (->> (if (<= (count repo-infos) 5)
         (mapv #(api-request authed-client
                             :get
                             (make-url "/projects" (str (:owner-login %) "/" (:repo-slug %)))
                             {})
               repo-infos)
         (get-user-projects authed-client true))
       (mapv (fn [{:keys [namespace visibility path id default_branch http_url_to_rep]}]
               {:fullPath [(:full_path namespace) path]
                :repository {:rootRef (or default_branch "master")}
                :httpUrlToRepo http_url_to_rep
                :namespace {:id (if (= "group" (:kind namespace))
                                  (group-id->gid (:id namespace))
                                  (format "gid://gitlab/User/%s" (:id namespace)))}
                :id (project-id->gid id)
                :visibility visibility}))))

(s/fdef repository-accessibility
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-infos (s/coll-of ::specs/repo-info)))
(defn repository-accessibility
  "Uses a GraphQL to check repo accessibility for a project."
  [authed-client repo-infos]
  (log/debugf "Finding repo availability for repositories %s" (pr-str repo-infos))
  (try+
    (let [projects (->> (get-filtered-projects authed-client (project-filter-terms repo-infos))
                        (map #(update % :fullPath split-path)))
          projects (if (seq projects) projects (repo-check-rest-api-adapter authed-client repo-infos))
          indexed (let [by-id (index-by :id projects)
                        by-name (index-by :fullPath projects)]
                    (fn [{:keys [external-id owner-login repo-slug]}]
                      (or (by-id external-id) (by-name [owner-login repo-slug]))))]
      (mapv
        (fn [repo-info]
          ;; IMPORTANT, if you do any changes here also fix adapter in repository-accessibility2
          (if-let [{:keys [fullPath httpUrlToRepo id namespace repository visibility]} (indexed repo-info)]
            (assoc repo-info
              :external-id id                               ; has to be set explicitly if searching by project names
              :accessible? true
              :private? (not= "public" visibility)
              :owner-login (first fullPath)
              :owner-type (if (some-> namespace :id (str/starts-with? "gid://gitlab/Group"))
                            "organization"
                            "user")
              :repo-slug (second fullPath)
              :clone-url httpUrlToRepo
              :default-branch (or (:rootRef repository) "master"))
            (assoc repo-info :accessible? false :private? true)))
        repo-infos))
   (catch [:type :http-error :status 401] _
     (log/errorf (:throwable &throw-context) "Error getting repository accessibility for %s" (pr-str repo-infos))
     repo-infos)))

(s/fdef get-web-hooks
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-infos (s/nilable (s/coll-of ::specs/repo-info))
                     :our-hook? ifn?
                     :concurrency int?))
(defn get-web-hooks
  [authed-client repo-infos our-hook? concurrency]
  (let [event-prop? #(str/ends-with? (name %) "_events")
        extract (fn [req-url {:keys [id url project_id created_at] :as res}]
                  {:external-id (project-id->gid project_id)
                   :url (str req-url "/" id)
                   :callback-url url
                   :events (reduce-kv (fn [acc k v] (if (true? v) (conj acc k) acc)) #{} (filter-keys event-prop? res))
                   :enabled? true
                   :timestamp created_at})
        run-fn (fn [paging-state]
                 (try+
                   (let [url (->> (repo-info->project-id paging-state)
                                  (format "/projects/%s/hooks")
                                  (request-url authed-client))
                         paging-state (api-client/with-item-xf paging-state (comp (map (partial extract url))
                                                                                  (filter our-hook?)))]
                     (api-page-request authed-client paging-state url {}))
                   (catch [:type :http-error :status 403] e
                     (log/infof "Could not access %s webhooks" (:repo-slug paging-state))
                     (assoc paging-state :cursor nil :items [] :error :inaccessible))))]
    (page/paginate! (page/async-fn run-fn concurrency) {} repo-infos)))


(s/fdef add-project-hook
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :url ::base-specs/uri
                     :token ::base-specs/non-empty-string))
(defn add-project-hook
  [authed-client repo-info url token]
  (let [resp
        (api-request*
          authed-client
          :post
          (format "/projects/%s/hooks" (repo-info->project-id repo-info))
          {:form-params {:url url
                         ;; these seem to be enabled unless you specifically disable them
                         :push_events false
                         :merge_requests_events true
                         :token token
                         :enable_ssl_verification true}})
        id (-> resp :body :id)]
    (api-client/to-url (:url resp) id)))

(s/fdef remove-project-hook
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :hook-id integer?))
(defn remove-project-hook
  [authed-client repo-info hook-id]
  (api-request
    authed-client
    :delete
    (format "/projects/%s/hooks/%s" (repo-info->project-id repo-info) hook-id)
    {}))

(s/fdef get-merge-request-refs
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :pr-id some?))
(defn get-merge-request-refs
  [authed-client repo-info pr-id]
  (loop []
    (let [{:keys [id diff_refs merge_status] :as resp}
          (api-request authed-client
                       :get
                       (format "/projects/%s/merge_requests/%s" (repo-info->project-id repo-info) pr-id)
                       {})
          ret (assoc (->kebab diff_refs) :id id)]
      (if (= merge_status "checking")
        (do (Thread/sleep 200) (recur))
        (do (when-not (-> ret :base-sha)
              (log/errorf "No base ref returned from GitLab! %s, %s" resp repo-info))
            ret)))))

(defn discussion->lead-note
  "Converts discussion with notes into the first note, but with :discussion-id key."
  [discussion]
  (let [lead-note (first (:notes discussion))]
    (assoc (select-keys (maps/->kebab lead-note) [:body :id :created-at :resolved :noteable-iid :position])
      :discussion-id (:id discussion)
      :has-replies? (< 1 (count (:notes discussion))))))

(s/def ::lead-note
  (s/keys :req-un [::body ::id ::created-at ::resolved ::discussion-id ::has-replies? ::noteable-iid]))

(s/fdef add-merge-request-discussion
        :args (s/cat :auth-token ::specs/auth-token
                     :external-id ::base-specs/non-empty-string
                     :pr-id integer?
                     :markdown-text ::base-specs/non-empty-string)
        :ret ::lead-note)
(defn add-merge-request-discussion
  [authed-client external-id pr-id markdown-text]
  (-> (api-request
        authed-client
        :post
        (format "/projects/%s/merge_requests/%s/discussions" (gid->id external-id) pr-id)
        {:form-params {:body markdown-text}})
      discussion->lead-note))

(s/fdef get-merge-request-lead-notes
        :args (s/cat :auth-token ::specs/auth-token
                     :external-id ::base-specs/non-empty-string
                     :pr-id integer?)
        :ret (s/coll-of ::lead-note))
(defn get-merge-request-lead-notes
  [authed-client external-id pr-id]
  (->> (get-all-pages
         authed-client
         (format "/projects/%s/merge_requests/%s/discussions" (gid->id external-id) pr-id)
         {:query-params {:order_by "created_at"}})
       (filter (complement :individual_note))
       (map discussion->lead-note)))

(s/fdef delete-merge-request-lead-note
        :args (s/cat :auth-token ::specs/auth-token
                     :external-id ::base-specs/non-empty-string
                     :lead-note ::lead-note))
(defn delete-merge-request-lead-note
  [authed-client external-id {:keys [noteable-iid discussion-id id]}]
  (api-request
    authed-client
    :delete
    (format "/projects/%s/merge_requests/%s/discussions/%s/notes/%s"
            (gid->id external-id) noteable-iid discussion-id id)
    {:unexceptional-status #(or (<= 200 % 299) (= 404 %))}))

(s/fdef edit-merge-request-lead-note
        :args (s/cat :auth-token ::specs/auth-token
                     :external-id ::base-specs/non-empty-string
                     :lead-note ::lead-note
                     :markdown-text ::base-specs/non-empty-string)
        :ret ::lead-note)
(defn edit-merge-request-lead-note
  [authed-client external-id {:keys [noteable-iid discussion-id id] :as note} markdown-text]
  (api-request
    authed-client
    :put
    (format "/projects/%s/merge_requests/%s/discussions/%s/notes/%s"
            (gid->id external-id) noteable-iid discussion-id id)
    {:form-params {:body markdown-text}})
  note)

(s/fdef resolve-merge-request-lead-note
        :args (s/cat :auth-token ::specs/auth-token
                     :external-id ::base-specs/non-empty-string
                     :lead-note ::lead-note
                     :resolved boolean?))
(defn resolve-merge-request-lead-note
  [authed-client external-id {:keys [noteable-iid discussion-id id]} resolved]
  (api-request
    authed-client
    :put
    (format "/projects/%s/merge_requests/%s/discussions/%s/notes/%s"
            (gid->id external-id) noteable-iid discussion-id id)
    {:query-params {:resolved resolved}}))

(defn get-user
  [authed-client]
  (-> (graphql-query authed-client (gql/generate-query {:currentUser [:id :name :username]}))
      (get-in [:body :data :currentUser])
      (update :id gid->id)))

(defn get-user-by-username
  "https://docs.gitlab.com/ee/api/graphql/reference/#queryuser"
  [authed-client username]
  (-> (graphql-query authed-client (gql/generate-query {`(user {:username ~username}) [:id :publicEmail]}))
      (get-in [:body :data :user])
      (update :id gid->id)
      (set/rename-keys {:publicEmail :email})))

(defn add-note-query
  "https://docs.gitlab.com/ee/api/graphql/reference/#mutationcreatediffnote"
  [{:keys [id head-sha base-sha _start-sha]} {:keys [path message old-line new-line old-path]} idx]
  (let [note-params {:body message
                     :noteableId (str "gid://gitlab/MergeRequest/" id)
                     ;start-sha must be base SHA for the comment to appear in files section; bug they have
                     :position (assoc-some {:headSha head-sha
                                            :startSha base-sha
                                            :baseSha base-sha
                                            :paths (assoc-some {} :oldPath old-path :newPath path)}
                                           :newLine new-line
                                           :oldLine old-line)}
        op (gql/as (str "m" idx) :createDiffNote)]
    {`(~op {:input ~note-params}) [:errors
                                   {:note {:id nil
                                           :discussion [:id]}}]}))
(defn resolve-discussion-query
  [discussion-id idx]
  (let [op (gql/as (str "m" idx) :discussionToggleResolve)]
    {`(~op {:input {:id ~discussion-id :resolve true}}) [:errors]}))

(defn add-notes-mutation
  [authed-client mr-specs notes]
  (let [statements (map-indexed (fn [idx note] (add-note-query mr-specs note idx)) notes)
        full-mutation (gql/generate-query {:mutation (apply merge statements)})
        resp (graphql-request authed-client full-mutation)
        data (-> resp :body :data)]
    (map-indexed (fn [idx note]
                   (let [{errors :errors {:keys [id discussion]} :note} (get data (keyword (str "m" idx)))]
                     (merge note {:errors (seq errors) :id id :discussion-id (:id discussion)})))
                 notes)))

(defn resolve-discussions
  "Uses graphql IDs"
  [authed-client inserted-notes]
  (doseq [ids-batch (->> inserted-notes (filter :resolve) (keep :discussion-id) (partition-all 100))]
    (try+
      (let [statements (map-indexed (fn [idx id] (resolve-discussion-query id idx)) ids-batch)
            full-mutation (gql/generate-query {:mutation (apply merge statements)})
            resp (graphql-request authed-client full-mutation)]
        (graphql-assert-success! resp))
      (catch [:status 500] e
        (log/info e "GitLab resolve discussions graphql mutation returned 500 response"))))
  inserted-notes)

(defn add-notes
  "Adds notes and resolves discussions as needed"
  [authed-client mr-specs notes]
  (->> (partition-all 5 notes)
       (mapcat #(try+
                  (add-notes-mutation authed-client mr-specs %)
                  (catch [:status 400] e
                    (log/info e "GitLab add notes graphql mutation returned 400 response")
                    %)
                  (catch [:status 500] e
                    (log/info e "GitLab add notes graphql mutation returned 500 response")
                    %)))))

(defn get-merge-requests
  [authed-client repo-info {:keys [since state] :as _search-options}]
  ;; TODO: refactor to use keywords and (?) assoc-some
  (let [query-params (merge {"per_page" pagelen
                             "order_by" "updated_at"
                             "sort"     "desc"
                             "scope" "all"}
                            (when state
                              {"state" state})
                            (when since
                              {"updated_after" (dates/date-time->string since)}))]
    (get-all-pages
     authed-client
     (format "projects/%s/merge_requests"
             (repo-info->project-id repo-info))
     {:query-params query-params})))

(defn get-labels [authed-client repo-info]
  (get-all-pages
    authed-client
    (format "projects/%s/labels" (repo-info->project-id repo-info))
    {}))

(defn get-issues [authed-client repo-info since]
  (get-all-pages
    authed-client
    (format "projects/%s/issues" (repo-info->project-id repo-info))
    {:query-params (merge {"scope" "all"}
                          (when since
                            {"updated_after" (dates/date-time->string since)}))}))

(defn get-boards
  [authed-client repo-info]
  (api-request
    authed-client
    :get
    (format "projects/%s/boards" (repo-info->project-id repo-info))
    {}))

(defn get-issue-label-events [authed-client repo-info iid]
  (api-request
    authed-client
    :get
    (format "/projects/%s/issues/%s/resource_label_events" (repo-info->project-id repo-info)
            iid)
    {}))

(defn gitlab-old-style-auth
  "See ticket 1688"
  [password]
  (reify api-client/ApiClient
    (augment-request [_this req] (assoc-in req [:headers "PRIVATE-TOKEN"] password))
    (on-expired-token [_this] nil)))

(defn host->api-url [host]
  (when (and host (not= host "gitlab.com"))
    (format "https://%s/api/v4" host)))

(comment
  (def repo-info {:repo-slug "analysis-target"
                  :owner-login "empear"})
  (def authed-client (gitlab-old-style-auth (System/getenv "GITLAB_TOKEN")))
  (get-merge-requests authed-client repo-info {:since (dates/string->date "2021-01-01")
                                               :state "merged"})
  )
