(ns codescene.features.repository-provider.gerrit.api
  (:require [clj-http.client :as http]
            [clj-http.util :as http-util]
            [ring.util.codec :as ring-codec]
            [clojure.spec.alpha :as s]
            [codescene.features.client.api :refer [to-url] :as api-client]
            [codescene.features.repository-provider.specs :as specs]
            [codescene.features.util.http-helpers :as h]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.features.util.string :refer [truncate]]
            [meta-merge.core :refer [meta-merge]]
            [clojure.string :as str]
            [taoensso.timbre :as log]
            [slingshot.slingshot :refer [try+]])
  (:import (java.io InputStream)))


(defn full-changes-id [project branch change-id]
  (format "%s~%s~%s" project branch change-id))

(defn- process-exception
  [exception {:keys [url] :as params}]
  (let [{:keys [body status content-type]} (ex-data exception)
        parsed-body (api-client/json-safe-parse body)
        message (or (and (= content-type :text/plain) body)
                    (get parsed-body :error_description)
                    (get-in parsed-body [:error :message])
                    (ex-message exception))
        data (map-of message url)]
    (log/debugf "Gerrit call returned %s %s" status body)
    (log/info exception "Gerrit call exception" (api-client/printable-req params))
    (case status
      401 (api-client/ex-unauthorized :gerrit exception data)
      403 (api-client/ex-forbidden :gerrit exception data)
      (api-client/ex-http-error :bitbucket exception (or status 0) data))))

(defn wrap-gerrit-magic
  "Handle the magix prefix that Gerrit prepends to the json body."
  [client]
  (let [json-content? (fn [resp] (-> resp
                                     (get-in [:headers "content-type"])
                                     http-util/parse-content-type
                                     :content-type
                                     (= :application/json)))
        skip-json-start (fn [_req resp]
                          (if (json-content? resp)
                            (update resp :body #(doto ^InputStream % (.skip 4)))
                            resp))]
    (fn
      ([req] (skip-json-start req (client req)))
      ([req response raise]
       (client req (fn [resp] (response (skip-json-start req resp))) raise)))))

(s/fdef api-request*
        :args (s/cat :auth-token ::specs/auth-token
                     :method #{:get :post :delete :put :patch :options :head
                               :trace :connect}
                     :url string?
                     :clj-http-params (s/nilable map?)))
(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :as :auto
                  :content-type :json
                  ;; gerrit returns non-gzipped responses even though
                  ;; it claims they are zipped
                  :decompress-body false
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url ""))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    (try
      (http/with-additional-middleware
        [wrap-gerrit-magic]
        (assoc (http/request final-params)
          :request-url (:url final-params)))
      (catch Exception e
        (-> e
            (process-exception final-params)
            (api-client/retry-token authed-client)
            (api-request* method url request-params))))))

(defn api-request [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(defn normalize-repo-path
  "When you call function like repo-url->parts you'll get different results in Gerrit depending on what
  protocols you use.

  When you have SSH URL like ssh://localhost:29418/Analysis-target, you get repo path of Analysis-target,
  but when you have HTTP URL like http://RokLenarcic@localhost:8082/a/Analysis-target you'll get a/Analysis-target.

  Normalize this."
  [repo-path] (str/replace-first repo-path #"^a?(/|$)" ""))

(defn- review-url [api-url project branch change-id revision-id]
  (h/full-url-for api-url (format "/changes/%s~%s~%s/revisions/%s/review"
                                  (ring-codec/url-encode project) branch change-id revision-id)))

(defn- webhooks-url [api-url repo-path]
  (h/full-url-for api-url (format "/config/server/webhooks~projects/%s/remotes"
                                  (ring-codec/url-encode repo-path))))

(defn delete-entity
  "Deletes an entity, returns true if successful"
  [authed-client entity-url]
  (api-request* authed-client
                :delete
                entity-url
                {:as :json-strict
                 :form-params {}})
  true)

(defn set-review
  [authed-client full-changes-id revision-id text comments]
  (api-request authed-client
               :post
               (to-url "/changes" full-changes-id "revisions" revision-id "review")
               {:form-params (cond-> {:tags "codescene" :message (truncate text 16383)}
                               comments (assoc :robot_comments comments))}))

(defn create-repo-webhook
  [authed-client repo-path url events]
  (let [form-params {:url url
                     :ssl_verify true
                     :connection_timeout 10000
                     :socket_timeout 10000
                     :retry_interval 10000
                     :max_tries 3
                     :events events}
        resp (api-request* authed-client
                           :put
                           (to-url "/config/server/webhooks~projects" repo-path "remotes/codescene")
                           {:form-params form-params})]
    {:id (-> resp :body :url)
     :url (:request-url resp)}))

(defn get-repo-webhook [authed-client {:keys [owner-login repo-slug]}]
  (try+
    (let [repo-path (normalize-repo-path (str owner-login "/" repo-slug))
          hook-url (to-url "/config/server/webhooks~projects" repo-path "remotes/codescene")
          {{:keys [events url]} :body rurl :request-url} (api-request* authed-client :get hook-url {})]
      {:events events
       :enabled? true
       :url rurl
       :callback-url url})
    (catch [:type :http-error] _
      nil)))

(defn get-repos-webhooks
  [authed-client repo-infos our-hook?]
  (keep #(if (and % (our-hook? %)) [%] []) (mapv #(get-repo-webhook authed-client %) repo-infos)))

(defn get-version
  [authed-client]
  (->> (api-request authed-client :get "/config/server/version" {})
       (re-seq #"\d+")
       (mapv parse-long)))
