(ns codescene.features.repository-provider.bitbucket-server.delta
  "See bitbucket delta docstring for explanation of builds and reports"
  (:require [clojure.set :as set]
            [clojure.string :as str]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.bitbucket.delta :as bb]
            [codescene.features.repository-provider.bitbucket-server.api :as api]
            [codescene.features.repository-provider.bitbucket.result :as result]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.url.url-utils :as url]
            [com.climate.claypoole :as cp]
            [medley.core :as m]
            [org.clojars.roklenarcic.paginator :as page]
            [taoensso.timbre :as log]))


(defn post-build
  "Add a pending build to PR and save link into provider-ref, requires access to
  source repository."
  [commit-info->client {:keys [pr-app-id pr-id target-branch commit-info result-options] :as provider-ref} {:keys [state text result-url]}]
  (let [app-id (str "codescene-" pr-app-id)]
    (when (:build? result-options)
      (log/debugf "post build, provider=bitbucket-server; action=delta-analysis; ref=%s" provider-ref)
      (api/create-build
        (commit-info->client commit-info)
        (:sha commit-info)
        (m/assoc-some {:name (format "CodeScene (%s) PR#%s" target-branch pr-id)
                       :key (cond-> app-id (< 40 (count app-id)) (subs 0 40))
                       :state state}
                      :url result-url
                      :description text)))))

(defn post-report
  "Unlike BitBucket Cloud the server doesn't support pending reports."
  [commit-info->client provider-ref {:keys [result details data link annotations]}]
  (let [{:keys [pr-app-id result-options commit-info target-branch]} provider-ref
        authed-client (commit-info->client commit-info)
        {:keys [report? annotations?]} result-options
        fix-link (fn [{:keys [text] :as v}]
                   (if text (set/rename-keys v {:text :linktext}) v))]
    ;; avoid pending report here
    (when-let [translated-status (and report? ({"FAILED" "FAIL" "PASSED" "PASS"} result))]
      (log/debugf "Update report, provider=bitbucket-server; action=delta-analysis; ref=%s" provider-ref)
      (log/debugf "Report annotations, provider=bitbucket-server; enabled=%s, #=%s" annotations? (count annotations))
      (api/create-report
        authed-client
        commit-info
        {:id (str "codescene-" pr-app-id)
         :title (format "CodeScene (%s)" target-branch)
         :details details
         :result translated-status
         :data (map #(update % :value fix-link) data)
         :reporter "CodeScene"
         :link link
         :logoUrl "https://codescene.io/cs-favicon.png"
         :annotations annotations}
        annotations?))))

(defn post-delta [commit-info->client provider-ref {:keys [comment build report]}]
  (bb/post-comment commit-info->client
                   {:add-pull-request-comment api/add-pull-request-comment
                    :delete-comments-with-runtag api/delete-comments-with-runtag}
                   provider-ref
                   comment)
  (post-build commit-info->client provider-ref build)
  (post-report commit-info->client provider-ref report))


(defn bb-server-repo->clone-url
  [repository]
  (->> repository :links :clone (filter #(= (:name %) "ssh")) first :href))

(defn full-name
  "Old BB versions don't have :links for repository, we cannot correctly parse repo-info from it. So
  to maintain kinda same code we fake it here"
  [{:keys [slug project]}] (str (:key project) "/" slug))

(defn full-name-ci
  "Lower cased version of BB repo full name, useful when comparing repository names derived from
  payloads with those derived from Clone URLs. Repository with full name AN/Analysis-target
  will have these URLs:

  HTTP urls:
  http://localhost:4000/projects/AN/repos/analysis-target/browse
  ssh://git@localhost:7999/an/analysis-target.git"
  [repository]
  (str/lower-case (full-name repository)))

(defn server-pr-endpoint
  "Parses PR endpoint and provider-def (:source or :destination) for branch, access token, commit-info, repo-name,

  Old BB versions don't have :links for repository, will use full-name->url function. C"
  [pr-endpoint full-name->url]
  (let [repo-name (-> pr-endpoint :repository full-name-ci)
        repository-url (or (->> pr-endpoint :repository bb-server-repo->clone-url)
                           (full-name->url repo-name))
        commit-info (-> repository-url
                        (url/repo-url->repo-info :bitbucket false)
                        (providers/commit-info (:latestCommit pr-endpoint)))]
    {:commit-info commit-info
     :repository-url repository-url
     :branch (some-> (:id pr-endpoint) providers/branch-ref->branch-name)
     :access nil
     :repo-name repo-name}))

(def ->provider-ref bb/->provider-ref)

(defrecord BitbucketServerDelta [commit-info->client]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta commit-info->client provider-ref (result/pending-update result-url)))
  (delta-results [this provider-ref pr-presentable]
    (->> (result/result-update pr-presentable result/change-item->server-annotation (:result-options provider-ref))
         (post-delta commit-info->client provider-ref)))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta commit-info->client provider-ref (result/skipped-update provider-ref reason result-url)))
  (delta-error [this provider-ref error result-url]
    (post-delta commit-info->client provider-ref (result/error-update error result-url))))

(defn webhook-status
  [pr-integration hooks]
  (cond
    (and (= 1 (count hooks))
         (every? :enabled? hooks)
         (every? #(pi/hook-token-verify pr-integration %) hooks))
    :installed
    (zero? (count hooks))
    :not-installed
    :else
    :invalid))

(defn repository-webhook-status
  [pr-integration repositories concurrency]
  (let [our-hook? (pi/our-hook-fn? pr-integration)
        authed-client (pi/-authed-client pr-integration (first repositories))
        ;; TODO Rok use authed client for each repo separately
        hooks (->> (api/get-repos-webhooks authed-client repositories our-hook? concurrency)
                   (mapv page/unwrap))]
    (mapv #(assoc {} :hooks %
                     :status (webhook-status pr-integration %)) hooks)))

(defrecord BitbucketServerHookAccess [concurr-limit]
  protocols/HookAccess
  (-repos-hooks [this PrIntegrationConfig repositories]
    (repository-webhook-status PrIntegrationConfig repositories concurr-limit))
  (-add-hooks [this PrIntegrationConfig repositories]
    (let [create #(api/create-repo-webhook (pi/-authed-client PrIntegrationConfig %1)
                                           %1
                                           (pi/-callback-url PrIntegrationConfig)
                                           ;; always just plain secret, since we are using onprem only
                                           (pi/-hook-secret PrIntegrationConfig)
                                           %2)]
      (cp/pdoseq concurr-limit
                 [repo repositories]
                 (log/infof "Installing Bitbucket webhook to %s" (:url repo))
                 (try
                   [(create repo ["pr:opened" "pr:modified" "pr:from_ref_updated"])]
                   (catch Exception _
                     ;; older Bitbucket Server versions don't have these events
                     [(create repo ["pr:opened" "repo:refs_changed"])])))))
  (-remove-hooks [this PrIntegrationConfig repos]
    (cp/pdoseq concurr-limit
               [repo repos
                :let [authed-client (pi/-authed-client PrIntegrationConfig repo)]
                hook (:hooks repo)]
               (log/infof "Removing Bitbucket webhook from %s" (:url hook))
               (api/api-request* authed-client :delete (:url hook) {}))))