(ns codescene.features.reports.code-health.pr-stats
  (:require
   [codescene.features.reports.pdf-helpers :as ph]
   [codescene.features.reports.graphics.functions :as graphics]
   [codescene.features.reports.common.pdf-elements :as pdf-elements]
   [codescene.features.reports.common.styling :as styling]))

(defn- empty-contents [img-dir pr-conf-url]
  (let [icon (styling/load-resource img-dir "empty-pr-statistics-icon.png")]
    (ph/table
      [] [1 6]
      (ph/row
        (ph/image [:prs-empty-image] icon)
        (ph/cell
          [:prs-empty-righthand-cell]
          (ph/paragraph [:prs-empty-title-paragraph] "Pull Request Integration")
          (ph/paragraph [:prs-empty-description-paragraph] "Enable CodeScene’s Pull Request Integration to run an automated analysis for each PR. Get feedback before merging code and see statistics such as the number of PR checks performed by CodeScene, findings detected, fixed, ignored and suppressed. If PR Integration is already enabled on this project and you are seeing this message, you may just need to run a new analysis.")
          [:spacer 1]
          (ph/table [] [16 47] (pdf-elements/link-row pr-conf-url "Configure PR Integration" img-dir)))))))

(defn- findings-text [percentage number]
  (let [f-text (if (= 1 number) "finding" "findings")]
    (format "%s%% (%s %s)" (int percentage) (int number) f-text)))

(defn- pr-statistics [{:keys [pull-requests negative-findings findings-fixed findings-ignored findings-suppressed]} img-dir]
  (let [fixed-percent (if (zero? negative-findings) 0 (Math/round (double (* 100 (/ findings-fixed negative-findings)))))
        ignored-percent (if (zero? negative-findings) 0 (Math/round (double (* 100 (/ findings-ignored negative-findings)))))
        suppressed-percent (if (zero? negative-findings) 0 (Math/round (double (* 100 (/ findings-suppressed negative-findings)))))
        fixed-text (findings-text fixed-percent findings-fixed)
        ignored-text (findings-text ignored-percent findings-ignored)
        suppressed-text (findings-text suppressed-percent findings-suppressed)]
    (ph/cell
      []
      (ph/table
        [:prs-righthand-table] nil
        (ph/row (ph/cell [:prs-righthand-cell]
                         (ph/paragraph [:c-dot-legend-paragraph] "PR Checks Performed")
                         (ph/paragraph [:prs-paragraph-large] (int pull-requests))))
        (ph/row (ph/cell [:prs-righthand-cell] (ph/line [:c-line]) [:spacer 1]))
        (ph/row (ph/cell [:prs-righthand-cell]
                         (ph/paragraph [:c-dot-legend-paragraph] "Findings Detected")
                         (ph/paragraph [:prs-paragraph-large] (int negative-findings))))
        (ph/row (ph/cell [:prs-righthand-cell] (ph/line [:c-dotted-line]) [:spacer 1]))
        (ph/row
          (ph/cell
            [:prs-righthand-cell]
            (ph/table
             [] nil
             (ph/row
              (pdf-elements/info-with-score-color :c-border-healthy "Fixed" fixed-text)
              (pdf-elements/info-with-score-color :c-border-risky "Ignored" ignored-text)
              (pdf-elements/info-with-score-color :c-border-suppressed "Suppressed" suppressed-text)))))))))


(defn- clean-up-value [value]
  (cond (number? value) (Math/round (double value))
        (string? value) (or (parse-double value) 0)
        :else 0))

(defn- clean-up-values [code-health]
  (let [pr-stat-keys [:pull-requests :negative-findings :findings-fixed :findings-ignored :findings-suppressed]
        default-values (zipmap pr-stat-keys (repeat 0))
        original-values (select-keys code-health pr-stat-keys)
        updated-values (update-vals original-values clean-up-value)]
    (merge default-values updated-values)))


(defn- contents [period-stats img-dir]
  (let [{:keys [negative-findings findings-fixed findings-ignored findings-suppressed] :as clean-period-stats} (clean-up-values period-stats)]
    (ph/table
      [] [3 10]
      (ph/row
        (ph/cell
          [:prs-graph-cell]
          (ph/table
            [:prs-graphics-table] nil
            (ph/row
              (ph/cell
                [:prs-graphics-cell]
                {:background-layer-fn
                 (ph/draw-using-g2d-fn
                   (graphics/pr-statistics-doughnut-fn
                     {:red-value   findings-ignored
                      :green-value findings-fixed
                      :grey-value findings-suppressed
                      :text        (format "%s Findings" negative-findings)}))}))))
        (pr-statistics clean-period-stats img-dir)))))


(defn pdf-element [{:keys [issues-detected issues-addressed issues-ignored pull-requests] :as period-stats} {:keys [img-dir pr-conf-url]}]
  (let [pr-stats-unavailable (every? styling/unavailable? [issues-detected issues-addressed issues-ignored pull-requests])]
    (ph/table
      [:c-card-content-top-table] nil
      (ph/row
        (ph/cell [] (ph/paragraph [:c-miniheader] "PULL REQUEST STATISTICS") [:spacer 2]))
      (ph/row
        (if pr-stats-unavailable
          (empty-contents img-dir pr-conf-url)
          (contents period-stats img-dir))))))
