(ns codescene.features.presentation.time-scope
  (:require [tick.core :as t]))

(defn- interval
  "Returns a vector of [start-time end-time] for the given period."
  [end-time period]
  [(t/<< end-time period) end-time])

(defn- interval-from-scope*
  "This should be considered the global definition of what we mean by the
  different scopes."
  [end-time scope]
  (case scope
    "week" (interval end-time (t/new-period 7 :days))
    "month" (interval end-time (t/new-period 30 :days))
    "year" (interval end-time (t/new-period 365 :days))
    (throw (ex-info "Unknown end-time scope" {:scope scope}))))

(defn interval-from-scope
  [end-time scope]
  #?(:clj  (interval-from-scope* end-time scope)
     ;; We're still figuring out js->clj interop with dates, for now we do the
     ;; conversion here. Currently we're expecting a js Date.toISOtring() as
     ;; end-time
     :cljs (let [end-instant (t/instant end-time)]
             (clj->js (interval-from-scope* end-instant scope)))))

(comment
  (interval-from-scope (t/now) "week") ; [#time/instant "2023-05-16T00:00:00Z" #time/instant "2023-05-24T00:00:00Z"]
  (interval-from-scope (t/now) "month") ; [#time/instant "2023-04-24T00:00:00Z" #time/instant "2023-05-24T00:00:00Z"]
  (interval-from-scope (t/now) "year")
  (interval-from-scope (t/now) "pineapple")
  (interval (t/now) (t/new-period 2 :days))
  (t/>> (t/truncate (t/now) :days) (t/new-period 1 :days))
  (t/truncate (t/now) :days))
