(ns codescene.features.pm-data.gitlab.gitlab-ticket-id-extractor
  "This namespace contains fns handling gitlab Issue id extraction and generic ticket id extraction 
   from gitlab PRs."
  (:require [codescene.features.pm-data.extractor-common :as ec]
            [codescene.url.url-utils :as url]))

;; Match either 
;; - issue numbers with an optional scope, like #123 or repo#123
;; - full issue URL:s, like 'https://gitlab.com/owner/repo/-/issues/123'
;; For issue numbers a relative repo/project scope will be matched (eg repo#123)
;; but scopes containing owner/group will not, (eg owner/repo#123)
;; See https://docs.gitlab.com/ee/user/project/issues/crosslinking_issues.html#from-commit-messages
;; for details about how GitLab supports crosslinking issues
;; This pattern is looks a bit hairy... what it does is
;; - use alternation (|) to match either the full scope of an issue URL (group 1) or the repo scope of an issue reference (group 2)
;; - for issue references, use negative lookbehind (?<!) to limit the scope to just the repo
;; - capture the issue number in group 3 
(def ticket-id-pattern #"(?:https://[^/]+/([\w/-]+)(?:/-/issues/)|(?<![/\w-])([\w-]+)?#)(\d+)")

(defn- ->scoped-ticket-id
  "Uses the capture groups from the match to decide on how to construct a scoped ticket id"
  [{:keys [owner-login repo-slug]} [match scope-from-url scope-from-ticket-id id]]
  (cond
    ;; Scope from issue url (match group 1) is full scope
    scope-from-url (format "%s#%s" scope-from-url id)
    ;; Scope from ticket id (match group 2) is just repo/project (thus only supported within the same group)
    scope-from-ticket-id (format "%s/%s#%s" owner-login scope-from-ticket-id id)
    :else (format "%s/%s#%s" owner-login repo-slug id)))

(defn extract-ticket-ids [repo-info msg]
  (->> msg
       (re-seq ticket-id-pattern)
       (map #(->scoped-ticket-id repo-info %))))

(defn make-pr-ticket-ids-lookup [provider-def repo-info]
  (ec/make-pr-ticket-ids-lookup provider-def (partial extract-ticket-ids repo-info)))

(defn make-ticket-id-extractor [provider-def repo-url]
  (let [repo-info (url/repo-url->repo-info* repo-url :gitlab false)
        ->pr-ticket-ids (make-pr-ticket-ids-lookup provider-def repo-info)]
    (fn [rev _date-time msg]
      (when repo-info
        ;; TODO: Use entire hash
        (concat (extract-ticket-ids repo-info msg)
                (->pr-ticket-ids (subs rev 0 7)))))))
