(ns codescene.features.pm-data.cache
  "Wraps some fetcher fns to make them cacheable"
  (:require [clj-time.core :as tc]
            [codescene.cache.core :as cache]
            [codescene.pm-data.provider-common :as common]
            [codescene.util.locks :as locks]
            [evolutionary-metrics.trends.dates :as dates]
            [medley.core :refer [distinct-by]]
            [memento.core :as m]
            [taoensso.timbre :as log]))

(defn- parse-time-str [s]
  (some-> s common/drop-ms-from-time-string dates/date-time-string->date))

(defn- unparse-time-str [s]
  (some-> s dates/date-time->string))

(defn make-cacheable
  "Make f usable with combine-with-cache.
   Requires f to be a fetcher function taking a since parameter as the first arg
   and returning a list of maps that has :id and :updated entries"
  [f]
  (fn [since & args]
    (let [items (apply f since args)]
      (with-meta
        items
        {:from (unparse-time-str since)}))))

(defn memo
  [fn-var key-fn]
  (cache/memo fn-var :pm-data key-fn))

(defn- full-data-requested-but-partial-data-cached
  [requested cached]
  (and (nil? requested) (some? cached)))

(defn- uncovered-partial-data-requested
  [requested cached]
  (and requested cached (tc/after? cached requested)))

(defn- now 
  "A fn to be memoized and used in combine-with-cache to make sure that calls during the same scope have the same notion of now."
  []
  (tc/floor (tc/now) tc/second)) ;; Note that we floor to second to match how it is stored in cache metadata.

(cache/memo-scoped #'now)

(defn combine-with-cache
  "Looks up data in the cache, decides what new data need to be fetched
   and calls the unwrapped function for new/updated data "
  [f key-fn since & more]
  (let [arg-list (list* since more)
        now      (now)]
    ;; Lock using the key-fn to make sure that simultaneous calls for the same cache value
    ;; results in a single fetch
    (with-open [_ (locks/lock-singleton! [f (apply key-fn arg-list)])]
      (let [data                     (apply f arg-list)
            {:keys [from to cached]} (meta data)
            from (parse-time-str from)
            to (parse-time-str to)
            unwrapped                (m/memo-unwrap f)
            items                    (cond
                                       ;; Use data as is on cache misses or when called with the same notion of now...
                                       (or (not cached)
                                           (= to now))
                                       data

                                       ;; ...or discard the cache data if unusable..
                                       (or (full-data-requested-but-partial-data-cached since from)
                                           (uncovered-partial-data-requested since from))
                                       (do
                                         (log/debugf "Cache value unusable - do a complete fetch!")
                                         (with-meta
                                           (apply unwrapped since more)
                                           {:from (unparse-time-str since)}))

                                       ;; ...or combine the two
                                       :else
                                       (do
                                         (log/debugf "Combine %d items in the cache with new/updated ones." (count data))
                                         (let [new-items (apply unwrapped (or to since) more)]
                                           (with-meta
                                             (->> (concat new-items data)
                                                  ;; distinct will skip stuff in data having the same id 
                                                  ;; (that is, older versions of the same thing)
                                                  (distinct-by :id))
                                             {:from (unparse-time-str from)}))))]
        (m/memo-add! f {arg-list (vary-meta (vec items) merge {:cached true
                                                               :to (dates/date-time->string now)})})
        items))))
