(ns codescene.features.pm-data.bitbucket.fetcher
  "Uses the BitBucket Server API for fetching data.
   This includes combining and transforming data to internal pm-data format, and in some cases catching and reporting errors."
  (:require [taoensso.timbre :as log]
            [codescene.features.repository-provider.bitbucket.api :as api]
            [codescene.url.url-utils :as url]
            [evolutionary-metrics.trends.dates :as dates]
            [slingshot.slingshot :refer [try+]]))


(defn- get-pr-commit-shas
  [authed-client repo-info id]
  (try+
   ;; It seems like bitbucket by design can return 404 in certain cases when fetching commits for a PR 
   ;; (see https://community.atlassian.com/t5/Bitbucket-questions/Getting-404-error-when-fetching-commit-data-despite-successfully/qaq-p/2492338)
   ;; Since 404 is changed to 403 in the api ns, let's just catch all errors... 
   (api/get-pr-commit-shas authed-client repo-info id)
   (catch [:type :http-error] {:keys [message]}
     (log/warnf "Failed to get commits for PR %s: %s" id message)
     [])))

(defn- pr->result [authed-client repo-info {:keys [description title id source merge_commit updated_on] :as _pr}]
  {:id (str id)
   :title title
   :branch (-> source :branch :name)
   :text description
   :merge-commit (-> merge_commit :hash)
   :commits (get-pr-commit-shas authed-client repo-info id)
   :updated (dates/timestamp->date updated_on)})

(defn fetch-pull-requests
  "Fetches merged pull-requests from the remote BitBucket API. Throws when the API calls fail."
  [since authed-client repo-info]
  (log/infof "Fetch pull requests from BitBucket since %s" (or since "-"))
  (let [search-options {:since since
                        :state "MERGED"}]
    (->> (api/get-pull-requests authed-client repo-info search-options)
         (mapv (partial pr->result authed-client repo-info)))))

(comment
  (def user (System/getenv "BITBUCKET_USER"))
  (def password (System/getenv "BITBUCKET_APP_PASSWORD"))
  (def api-url "https://api.bitbucket.org/2.0")
  (def repo-info (codescene.url.url-utils/repo-url->repo-info  "git@bitbucket.org:empear/analysis-target.git" :bitbucket true))
  (fetch-pull-requests (dates/string->date "2021-07-04") [user password] repo-info)
  )
