(ns codescene.features.factors.api
  (:require
    [clojure.string :as str]
    [clj-time.core :as t]
    [clj-time.format :as f]
    [codescene.factors.core :as factors]
    [codescene.factors.delivery.presentation :as delivery]
    [codescene.factors.knowledge.presentation :as knowledge]
    [codescene.factors.team-code.presentation :as tca]
    [codescene.features.api.core :as api-core]
    [codescene.features.components.analysis :as analysis]))

;: BEGIN PUBLIC DANGER FUNCTIONS
;; We'll create a contract with our customers with the data model we let out of
;; here, so let's be careful and hide as much as possible.

;; We'll start with the hacky version of selecting things from the time series,
;; and then over time propagate our refactoring further down the stack towards
;; the database, making sure that the UI also gets simpler code.



(defn- after-or-equal? [date from-date]
  (or (t/equal? date from-date)
      (t/after? date from-date)))

(defn- before-or-equal? [date end-date]
  (or (t/equal? date end-date)
      (t/before? date end-date)))

(defn- clean-up-sample [{:keys [date kpi] :as _sample}]
  {:date (first (str/split date #"T"))
   :kpi  (double kpi)})

;; made public to allow for unit test
(defn clean-up-trend [trend]
  (->> trend
       (remove (fn [{kpi :kpi}] (nil? kpi)))
       (map clean-up-sample)))

(defn trend-in-date-range [start-date end-date raw-trend]
  (let [filter-fn (cond
                    (and start-date end-date) (fn [date] (t/within? start-date end-date date))
                    start-date (fn [date] (after-or-equal? date start-date))
                    end-date (fn [date] (before-or-equal? date end-date)))]
    (filter (fn [sample] (let [parsed-sample-date (f/parse (:date sample))]
                           (filter-fn parsed-sample-date))) raw-trend)))

(defn filter-trend [trend start-date end-date]
  (if (or start-date end-date)
    (let [filtered-trend (trend-in-date-range start-date end-date trend)]
      (clean-up-trend filtered-trend))
    (clean-up-trend trend)))



(defn- extract-sub-kpi-value [sub-kpi-key {:keys [sub-kpis] :as _sample}]
  (let [sub-kpi (first (filter (fn [{k :key}]
                                 (= (name sub-kpi-key) k)) sub-kpis))]
    (get sub-kpi :value)))
(defn- extract-sub-kpi-presentation-value [sub-kpi-key {:keys [sub-kpis] :as _sample}]
  (let [sub-kpi (first (filter (fn [{k :key}]
                                 (= (name sub-kpi-key) k)) sub-kpis))]
    (:value (:presentation sub-kpi))))


(defn- select-sub-kpi-from-sample [sub-kpi-key sub-kpi-fn {:keys [date] :as knowledge-sample}]
  {:date date
   :kpi  (sub-kpi-fn sub-kpi-key knowledge-sample)})

(defn select-sub-kpi-from-trend [sub-kpi-key trend]
  (map (partial select-sub-kpi-from-sample sub-kpi-key extract-sub-kpi-value) trend))

(defn select-sub-kpi-presentation-value-from-trend [sub-kpi-key trend]
  (map (partial select-sub-kpi-from-sample sub-kpi-key extract-sub-kpi-presentation-value) trend))


(defn- flatten-delivery-sample [{:keys [date kpis] :as _sample}]
  (let [{{:keys [descriptions kpi sub-kpis]} :month} kpis]
    {:date date
     :descriptions descriptions
     :kpi kpi
     :sub-kpis sub-kpis}))
(defn flatten-delivery-trend [trend]
  (map flatten-delivery-sample trend))


(defn latest-raw-trend [main-kpi-key system project-id]
  (let [analysis (api-core/api-analyses system)
        latest-analysis-id (:id (analysis/latest-successful-analysis analysis project-id))
        path ((analysis/local-analysis-result-path analysis project-id latest-analysis-id) "")]
    (factors/time-series-by-type path main-kpi-key)))

(defn with-date-filter [start end fn]
  (let [start-date (f/parse start)
        end-date (f/parse end)]
    (fn start-date end-date)))



(comment
  (def raw-knowledge-trend
    (factors/time-series-by-type
      "/home/ulrikawiss/cloudpath/codescene-cloud-results/113/4af3008b-dfcb-4572-9a68-012e047f363a/results/analysis20221028033901"
      ;"/home/aasa/data/cs/cloud/308/3eea2a88-1d9c-426f-a7d4-21014a4230b3/results/analysis20221025084454"
      knowledge/kpi-key))
  (def knowledge-sample (first raw-knowledge-trend))
  (select-code-familiarity-from-trend raw-knowledge-trend)
  (filter-trend raw-knowledge-trend knowledge/kpi-key nil nil)

  (def raw-delivery-trend
    (factors/time-series-by-type
      "/home/ulrikawiss/repos/codescene/features/test/codescene/features/factors/results_test_data/delivery_factor_with_duplicates"
      ;"/home/ulrikawiss/cloudpath/codescene-cloud-results/113/4af3008b-dfcb-4572-9a68-012e047f363a/results/analysis20221028033901"
      ;"/home/aasa/data/cs/cloud/308/3eea2a88-1d9c-426f-a7d4-21014a4230b3/results/analysis20221025084454"
      delivery/kpi-key))

  (def flattened-delivery-trend (flatten-delivery-trend raw-delivery-trend :month))
  (def sub-kpi-trend (select-sub-kpi-from-trend delivery/unplanned-work-key flattened-delivery-trend))

  (def sub-kpi-presentation-value
    (extract-sub-kpi-presentation-value
      delivery/unplanned-work-key
      [{:key "development-time",
        :name "Development Time",
        :value 6.77,
        :presentation {:unit "minutes", :value 2167.5},
        :completed-tasks 34,
        :value-in-minutes 2167.5}
       {:key "amount-of-unplanned-work",
        :name "Amount of Unplanned Work",
        :value 9.91,
        :presentation {:unit "percent", :value 0},
        :total-tickets-worked-on 34}]))


  (def raw-tca-trend
    (factors/time-series-by-type
      "/home/ulrikawiss/cloudpath/codescene-cloud-results/129/707893fb-c44d-4ba8-830f-e596c914cf77/results/analysis20221106215400"
      ;"/home/ulrikawiss/cloudpath/codescene-cloud-results/113/4af3008b-dfcb-4572-9a68-012e047f363a/results/analysis20221028033901"
      ;"/home/aasa/data/cs/cloud/308/3eea2a88-1d9c-426f-a7d4-21014a4230b3/results/analysis20221025084454"
      tca/kpi-key))

  (def tca-team-cohesion-trend
    (select-sub-kpi-from-trend tca/team-cohesion-key raw-tca-trend))

  (def filtered-tca-trend
    (filter-trend tca-team-cohesion-trend nil nil))
  )

