(ns codescene.features.delta.webhooks
  (:require [buddy.core.codecs :as codecs]
            [buddy.core.mac :as mac]
            [clojure.string :as str]
            [ring.util.response :refer [response status]]
            [taoensso.timbre :as log])
  (:import (org.bouncycastle.util.encoders DecoderException)))

(def responses
  {:azure {:unconfigured (status (response "Disable") 410)}})

(defn wrap-error-responses [handler provider-id]
  (fn [req]
    (try
      (handler req)
      (catch Exception e
        (let [intent (-> e ex-data :delta/hook-intent)]
          (when-not intent
            (log/error e (format "Webhook error %s %s" provider-id (:uri req))))
          (get-in responses [provider-id intent] (response "OK")))))))

(defn valid-github-sig?
  "Checks that the signature represented by the given hash
  corresponds to a proper signature for given body."
  [secret body signature]
  (try
    (let [signature (some-> signature (str/replace-first "sha1=" ""))
          options {:key (if (empty? secret) "_" secret) :alg :hmac+sha1}
          ok? (mac/verify body (codecs/hex->bytes signature) options)]
      (when-not ok?
        (log/debugf "Signature mismatch, theirs: %s, ours: %s" signature (codecs/bytes->hex (mac/hash body options))))
      ok?)
    (catch DecoderException e
      (log/infof "Signature decoding error: %s" e)
      false)))
