(ns codescene.features.delta.protocols)

(defprotocol DeltaResultBoundary
  "Gateway protocol for Posting Delta Results to Provider CI systems.

  Most Git Server providers have a general concept of CI check. The actual design
  differs in providers.

  To interact with check runs in different providers, protocol implementations will need
  different info, and provider reference stores such information, usually in the JSON form.

  The result URL is an absolute URL.

  Provider reference is opaque outside the protocol implementations, and it should be treated as such.

  Provider reference is limited to 2kb."
  (delta-pending [this provider-ref result-url]
    "Creates or updates CI check into pending status.

    provider-ref is a provider specific map used by implementor to locate/manipulate
    the check in the provider's systems
    result-url is the absolute URL of the delta result in CodeScene")
  (delta-results [this provider-ref presentable-result]
    "Updates the CI check with the results of delta analysis.

    provider-ref is a provider specific map used by implementor to locate/manipulate
    the check in the provider's systems
    result-url is the absolute URL of the delta result in CodeScene")
  (delta-skipped [this provider-ref reason result-url]
    "Updates the CI check with a status that indicates skipped check, and the reason why it was skipped.

    provider-ref is a provider specific map used by implementor to locate/manipulate
    the check in the provider's systems
    result-url is the absolute URL of the delta result in CodeScene")
  (delta-error [this provider-ref error result-url]
    "Updates CI check with the error.

    provider-ref is a provider specific map used by implementor to locate/manipulate
    the check in the provider's systems
    result-url is the absolute URL of the delta result in CodeScene"))

(defprotocol CIProvider
  (ci-installed? [this PrIntegrationConfig repositories]
    "Returns about each repo:
     - `:ci-installed?` true/false depending on the status
     - `:ci-status-string` a string describing the status in more detail, can be nil

     As minimum, the repo data should contain `:url` and owner information.")
  (enable-ci-integration! [this PrIntegrationConfig repositories]
    "Called when PR integration is enabled. also called when a repository is added to a project.")
  (disable-ci-integration! [this PrIntegrationConfig repositories]
    "Called when a project is removed, a repository is removed from project or PR integration is disabled."))

(defprotocol HookAccess
  "Object to give hook access to config process"
  (-repos-hooks [this PrIntegrationConfig repositories] "Hooks for repos. For each input repo it returns a map of:
  - :hooks vector of hooks
  - :status one of :installed, :not-installed, :inaccessible, :invalid
  - :status-string")
  (-add-hooks [this PrIntegrationConfig repositories] "Add hooks, returns a coll of hook URLs for each repo installed")
  (-remove-hooks [this PrIntegrationConfig repositories] "Remove hooks, that are under :hooks key in repository"))