(ns codescene.features.delta.integration.integration-db
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.util.coll :refer [do-diff]]
            [codescene.features.util.maps :refer [map-of-db map-of map-of-kebab? ->db-keys]]
            [codescene.features.config.presets :as presets]
            [codescene.features.config.properties :as p]
            [hugsql.core :as hugsql]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(hugsql/def-db-fns "codescene/features/delta/integration/integration.sql")

(defn empty-config
  [project-id provider->properties provider-id]
  (let [properties (provider->properties provider-id)]
    (p/config-set {:project-id project-id
                   :provider-id provider-id
                   :ch-enabled? false
                   :cov-enabled? false}
                  (p/from-db (provider->properties provider-id) {})
                  properties
                  :live)))

(defn- get-conf-kv
  "Get kv-map with keyword keys and string values. "
  [db-spec project-id]
  (into {} (map (juxt (comp keyword :setting_key)
                      :setting_value))
        (select-kv db-spec (map-of-db project-id))))

(defn- resolve-global-preset
  "Try to find global preset"
  [tx default category scope]
  (if-let [id (get-in (presets/available-settings tx scope [category])
                      ["cov-config" 0 :id])]
    (presets/get-settings tx id scope)
    default))

(defn- preset-type [v]
  (cond
    (= "global" v) :global
    (integer? v) :by-id
    :else :none))

(defn preset-desc
  "Returns Preset Description.

  - category (required): preset category
  - post-proc-fn: a function data' = fn(data)
  - default: if preset is global and none is defined use this."
  [category post-proc-fn default]
  (map-of category post-proc-fn default))

(defn resolve-presets
  "Look at the key and resolve the presets. k->category map
  should contain a property key to preset category.

  k->preset-desc is a map from property key to preset-desc

  Returns config-set with data having resolved keys, and metadata containing
  a map with resolved keys. It will not resolve keys that have been resolved
  again."
  [config-set db-spec k->preset-desc scope]
  (if (some (:data config-set) (keys k->preset-desc))
    (jdbc/with-db-transaction [tx db-spec]
      (reduce-kv
        (fn [config-set k {:keys [category post-proc-fn default] :or {post-proc-fn identity}}]
          (let [preset-id (get-in config-set [:data k])
                ;; select preset
                preset (case (preset-type preset-id)
                         :global (resolve-global-preset tx default category scope)
                         :by-id (presets/get-settings tx preset-id scope)
                         nil)]
            ;; run post processing fn
            (if preset
              (-> config-set
                  (update :data #(merge % (-> preset :kv post-proc-fn)))
                  (assoc-in [:metadata :presets-loaded k] (dissoc preset :kv :uses)))
              config-set)))
        config-set
        ;; resolve presets minus those that have already been loaded
        (apply dissoc k->preset-desc (-> config-set :metadata :presets-loaded keys))))
                config-set))

(defn update-core-configuration
  "Updates/inserts project-id, ch-enabled?, cov-enabled?"
  [db-spec data]
  (jdbc/with-db-transaction [tx db-spec]
    (let [data (->db-keys (m/update-existing data :provider-id name))
          prev (select-by-project-id tx data)]
      (if prev
        (update-core! tx data)
        (when (:project_id data)
          (insert-core! tx (update data :provider_id #(or % "unknown"))))))))

(defn delete [tx project-id]
  (jdbc/execute! tx ["delete from project_delta_config where project_id = ?" project-id])
  (jdbc/execute! tx ["delete from project_delta_config_kv where project_id = ?" project-id]))

(defn- update-configuration*
  "Assumes that config-set is in :db state."
  [db-spec {:keys [properties data metadata]}]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [{:keys [project-id]} metadata
          ;; old data, only keep keys from properties to not add/delete outside the scope
          ;; of the update
          old-data (select-keys (into {} (map (juxt :setting_key :setting_value))
                                      (select-kv db-spec (map-of-db project-id)))
                                (map (comp str symbol :id) properties))
          {:keys [removed added updated]} (do-diff first old-data data)]
      (update-core-configuration tx metadata)
      (doseq [[setting-key _] removed]
        (delete-key! tx (map-of-db project-id setting-key)))
      (doseq [[setting-key setting-value] added]
        (insert-kv! tx (map-of-db project-id setting-key setting-value)))
      (doseq [[_old [setting-key setting-value]] updated]
        (update-kv! tx (map-of-db project-id setting-key setting-value))))
    (:project-id metadata)))

(defn update-configuration
  "Updates configuration, only considering the properties given. In case
  where providers same space e.g. bitbucket vs github delta, provide the
  union of all possible properties, so that in case of provider type switching, the old
  provider data is all deleted."
  [db-spec {:keys [state] :as config-set}]
  (case state
    :ui-params (throw (ex-info "Cannot call this function with UI Params" {}))
    :live (recur db-spec (p/live-to-db config-set))
    :db (update-configuration* db-spec config-set)))

(defn get-core-config
  [tx project-id]
  (let [{:keys [project_id ch_enabled cov_enabled provider_id]} (select-by-project-id tx {:project_id project-id})]
    (when project_id
      (map-of-kebab? project_id ch_enabled cov_enabled (keyword provider_id)))))

(defn get-all-core-configs
  [tx filter]
  (select-all tx (->db-keys filter)))

(defn get-config
  "Returns live config set with these properties."
  [tx project-id provider->properties]
  (let [{:keys [provider-id] :as core} (get-core-config tx project-id)
        properties (provider->properties provider-id)]
    (when core
      (p/config-set
        core
        (p/from-db properties (get-conf-kv tx project-id))
        (or properties [])
        :live))))

(defn get-all-configs-for-provider
  "Returns all configs of a certain provider"
  [tx provider-id properties]
  (let [projects (group-by :project_id (select-by-provider tx (map-of-db (name provider-id))))
        ->config-set (fn [[{:keys [project_id ch_enabled cov_enabled provider_id]} :as all]]
                       (p/config-set
                         (map-of-kebab? project_id ch_enabled cov_enabled (keyword provider_id))
                         (p/from-db properties
                                    (into {} (map (juxt (comp keyword :setting_key)
                                                        :setting_value)) all))
                         properties
                         :live))]
    (-> projects
        (update-vals ->config-set)
        vals
        vec)))