(ns codescene.features.components.mock-http
  (:require [buddy.core.codecs :as codecs]
            [clojure.edn :as edn]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [codescene.util.data :as data]
            [medley.core :as m]
            [ring.util.codec :as codec]
            [taoensso.timbre :as log]
            [clj-http.conn-mgr :as conn]
            [clj-http.core :as http-core]
            [codescene.features.util.maps :refer [map-of]])
  (:import (org.apache.http HttpEntity HttpHost HttpRequest HttpEntityEnclosingRequest HttpMessage Header HttpResponse ProtocolVersion)
           (org.apache.http.concurrent BasicFuture FutureCallback)
           (org.apache.http.impl.client CloseableHttpClient)
           (org.apache.http.client.methods CloseableHttpResponse HttpUriRequest)
           (org.apache.http.entity ByteArrayEntity ContentType)
           (java.io ByteArrayOutputStream)
           (org.apache.http.impl.nio.client CloseableHttpAsyncClient)
           (org.apache.http.message BasicHeader BasicHttpResponse)
           (org.apache.http.protocol HttpContext)))

(defn- read+clone-entity
  "HttpEntity has an InputStream, which if read, then throws exception the next time it's read.
  So we must create a ne w entity when we read it, so it can be set on the req/resp for middleware up the
  chain to work. Returns pair base64-content"
  [^HttpEntity entity set-entity-fn]
  (when entity
    (let [os (ByteArrayOutputStream.)]
      (io/copy (.getContent entity) os)
      (set-entity-fn (ByteArrayEntity. (.toByteArray os) (ContentType/get entity)))
      (-> (.toByteArray os) data/b64-urlsafe))))

(defn- headers [^HttpMessage m ignore-header?-fn]
  (reduce (fn [m ^Header header]
            (if (ignore-header?-fn (str/lower-case (.getName header)))
              m
              (assoc m (.getName header) (.getValue header))))
          {}
          (.getAllHeaders m)))

(defn- parse-resp [^HttpResponse resp]
  (let [entity (.getEntity resp)
        status (.getStatusLine resp)
        protocol-version (.getProtocolVersion status)]
    {:status (.getStatusCode status)
     :headers (headers resp (constantly false))
     :body (read+clone-entity entity #(.setEntity resp %))
     :protocol-version {:name (.getProtocol protocol-version)
                        :major (.getMajor protocol-version)
                        :minor (.getMinor protocol-version)}
     :reason-phrase (.getReasonPhrase status)}))

(defn- ->basic-resp [{:keys [status reason-phrase protocol-version body headers]}]
  (let [resp (proxy [BasicHttpResponse CloseableHttpResponse]
                    [(ProtocolVersion. (:name protocol-version) (:major protocol-version) (:minor protocol-version))
                     ^Integer status
                     ^String reason-phrase]
               (close []))]
    (doseq [[k v] headers]
      (.addHeader resp k v))
    (when body
      (.setEntity resp (ByteArrayEntity. (-> body codecs/str->bytes (codecs/b64->bytes true)))))
    resp))

(defn mock-file-name [req {:keys [file-prefix ignored-headers ignored-query-params] :as _mock-rules}]
  (let [ignore-header?-fn (into #{"connection"} (map name) ignored-headers)
        uri (.getURI req)
        body (when (instance? HttpEntityEnclosingRequest req)
               (read+clone-entity (.getEntity ^HttpEntityEnclosingRequest req)
                                  #(.setEntity ^HttpEntityEnclosingRequest req %)))
        data {:method (.getMethod req)
              :uri {:scheme (.getScheme uri)
                    :path (.getPath uri)
                    :query-params (when-let [uri (.getRawQuery uri)]
                                    (apply dissoc (codec/form-decode uri) (map name ignored-query-params)))
                    :fragment (.getFragment uri)}
              :body body
              :headers (headers req ignore-header?-fn)
              :protocol-version (.getProtocolVersion req)}
        mock-file-name (str file-prefix "_" (hash data) ".edn")]
    (log/infof "Generated mock file name %s from %s" mock-file-name (pr-str data))
    mock-file-name))

(defn reading-client
  "Reads responses from EDN files based on request properties.

  mock-rules is keys file-prefix ignored-headers ignored-query-params"
  [req-to-filename base-opts]
  (let [client (proxy [CloseableHttpClient] []
                 (close [] nil)
                 (doExecute [_host ^HttpRequest req _ctx]
                   (let [file-name (req-to-filename req)]
                     (log/infof "Reading mocked request file %s" file-name)
                     (->basic-resp (edn/read-string (slurp file-name))))))
        async-client (proxy [CloseableHttpAsyncClient] []
                       (isRunning [] true)
                       (start [] nil)
                       (close [] nil)
                       ;; damn overrides, now we need to hack this. Ok so we know which signature clj-http is calling
                       ;; and we can just implement that
                       (execute [^HttpUriRequest req ^HttpContext context ^FutureCallback callback]
                         (let [file-name (req-to-filename req)]
                           (log/infof "Reading mocked request file %s" file-name)
                           (try
                             (doto (BasicFuture. callback)
                               (.completed (->basic-resp (edn/read-string (slurp file-name)))))
                             (catch Exception e
                               (.failed callback e)
                               (throw e))))))]
    (merge base-opts
           {:http-client client
            :connection-manager (conn/make-reusable-conn-manager {})
            :async-http-client async-client})))

(defn ^HttpUriRequest map-auth-tokens
  "Map authorization header, substitutes a mock header with a real one, when recording requests"
  [^HttpUriRequest request auth-header-mapping]
  (let [mapping (or auth-header-mapping {})]
    (some->> (.getFirstHeader request "authorization")
             (.getValue)
             mapping
             (BasicHeader. "authorization")
             (.setHeader request)))
  request)

(defn recording-client
  "Records all request data into a series of EDN files."
  [req-to-filename base-opts auth-header-mapping]
  (let [cm (conn/make-reusable-conn-manager {})
        ^CloseableHttpClient base-client (http-core/build-http-client {} true cm)
        client (proxy [CloseableHttpClient] []
                 (close [] (.close base-client))
                 (doExecute [^HttpHost host ^HttpRequest req, ^HttpContext ctx]
                   (let [file-name (req-to-filename req)]
                     (log/debugf "Writing request data to file %s" file-name)
                     (let [resp (.execute base-client host (map-auth-tokens req auth-header-mapping) ctx)]
                       (->> (parse-resp resp) pr-str (spit file-name))
                       resp))))
        acm (conn/make-reusable-async-conn-manager {})
        ^CloseableHttpAsyncClient base-async-client (http-core/build-async-http-client {} acm)
        async-client (proxy [CloseableHttpAsyncClient] []
                       (isRunning [] (.isRunning base-async-client))
                       (start [] (.start base-async-client))
                       (close [] (.close base-async-client))
                       ;; damn overrides, now we need to hack this. Ok so we know which signature clj-http is calling
                       ;; and we can just implement that
                       (execute [^HttpUriRequest req ^HttpContext context ^FutureCallback callback]
                         (let [file-name (req-to-filename req)
                               cb (reify FutureCallback
                                    (completed [this v]
                                      (->> (parse-resp v) pr-str (spit file-name))
                                      (.completed callback v))
                                    (failed [this e] (.failed callback e))
                                    (cancelled [this] (.cancelled callback)))]
                           (log/debugf "Writing request data to file %s" file-name)
                           (.execute base-async-client (map-auth-tokens req auth-header-mapping) context cb))))]
    (merge {:socket-timeout 30000
            :connection-timeout 30000}
           base-opts
           {:connection-manager cm
            :http-client client
            :async-conn-manager acm
            :async-http-client async-client})))

(defn mocked-client [{:keys [file-prefix record-requests?
                             ignored-headers
                             ignored-query-params
                             auth-header-mapping
                             base-opts]}]
  (let [req-to-filename #(mock-file-name % (map-of file-prefix ignored-headers ignored-query-params))]
    (if record-requests?
      (recording-client req-to-filename base-opts auth-header-mapping)
      (reading-client req-to-filename base-opts))))
