(ns codescene.features.code-coverage.gates.executable-loc-estimator
  "As part of our coverage report, we want to sum up the amount 
   of uncovered lines of code. To do this, we:
    1. Run the X-Ray to figure our what functions we have.
       - Code that is outside of a function is deemed non-executable.
   
    2. Run loco on each function.
       - Strip out lines with comments.
   
    3. The resulting set is a mapping from function_name -> executable_loc_candidates.
       - We say 'candidates' because we cannot tell for sure yet; we also need to coverage 
         report in order to piece it all together."
  (:require [evolutionary-metrics.mining.file-patterns :as patterns]
            [codescene.features.code-coverage.gates.code-lines-detector :as detector]
            [hotspots-x-ray.recommendations.code-extraction :as code-extraction]
            [hotspots-x-ray.languages.parser :as parser]
            [evolutionary-metrics.complexity.loco-rules :as loco-rules]
            [evolutionary-metrics.complexity.loco :as loco]
            [com.climate.claypoole :as cp]
            [clojure.string :as s]
            [taoensso.timbre :as log]))

(defn- language-rules-for
  [file-path]
  (-> file-path patterns/extension-of loco-rules/rule-for))

(defn- code-field
  [[_blanks _comments code]]
  code)

(defn- lines-in
  [a-file]
  (-> a-file
      slurp 
      s/split-lines))

(defn- active-code-size-in
  [a-file]
  (if-let [rules (language-rules-for a-file)]
    (-> (lines-in a-file)
        (loco/code-stats-in rules) 
        code-field)
    0))

(defn- with-loc-per-function
  [parsed-fns]
  (->> parsed-fns
       (map (fn [{:keys [name start-line end-line]}]
              [name (set (range start-line (inc end-line)))]))
       (into {})))

(defn- raw-functions-in
  [content parsed-functions]
  (let [lines-in-file (s/split-lines content)]
    (code-extraction/code-by-function-name lines-in-file parsed-functions)))

(defn- ->executable-lines-per-function
  [executable-candidate? parsed-fns]
  (->> parsed-fns
       (map (fn [{:keys [name start-line end-line body]}]
              (let [all-lines (range start-line (inc end-line))
                    code-lines (filter executable-candidate? all-lines)]
                [name code-lines])))
       (into {})))

(defn- executable-candidate-lines-in
  [file-path]
  (if (parser/supports-file-type? file-path)
    (try
      (let [file-content (slurp file-path)
            parsed-functions (->> file-content
                                  (parser/parse-function-statistics file-path)
                                  (raw-functions-in file-content))
            executable-candidate? (set (detector/find-non-commented-lines-in file-path file-content))]
        (->executable-lines-per-function executable-candidate? parsed-functions))
      (catch Exception ex
        (log/error ex "cannot get executable candidate lines in: " file-path)
        {}))
    {}))

(defn augment-files-with-lines-of-code
  [file-paths]
  ; let's run the LoC calculation on all available threads; we run within a build context, so 
  ; the machine should be dedicated to this:
  (let [ideal-threads (* (cp/ncpus) 2) ; assume some IO bound, so * 2.
        thread-pool   (cp/threadpool (max ideal-threads 1))] ; we'll deadlock without any threads :/
    (cp/pmap thread-pool
             (fn [f]
               (let [executable-by-fn (executable-candidate-lines-in f)
                     active-code-size (->> executable-by-fn
                                           vals
                                           (map count) ; sum the number of exectuable lines for each fn
                                           (reduce +))]
                 {:file-name f
                  :active-code-size active-code-size
                  :executable-candidate-lines executable-by-fn}))
             file-paths)))