(ns codescene.features.code-coverage.gates.criteria
  "This namespace contains the business rules for all supported coverage gates.
   
   The input is driven by config data + a coverage report on our internal format. In addition, 
   some gates might require a diff (e.g. new_and_changed_code).

    - The config data is described here: https://codescene.slab.com/posts/code-coverage-quality-gates-cc-gq-in-pull-requests-pr-68mr7ciy
   
    - Specific coverage reports get converted to and internal and generic format. Here's what it looks like:
      
      ({:path ui/preview/examples/payment-page/countries.js, 
        :name countries.js, 
        :line-coverage {
           :covered 1252, 
           :total 1252, 
           :coverage 1
           ; the following detailed line-by-line coverage is unique to the 'check' CLI command and 
           ; won't be part of the 'upload' used in the full analysis:
           :details {
             :covered #{1 10 11 12 120 ...}
             :uncovered #{ 4 5 6 ...}}
           }} 
      {:path ui/preview/examples/payment-page/us-states.js, 
       ...
      }})
   
      NOTE: line-coverage is always included. Other metrics might exist too.

    - The diff (typically for a PR) has the following format:
      
      ({:file-name 'ui/…../a.clj'
        :lines #{1 3 6 7 ...}}
       {:file-name 'analysis/src/core.clj'
        :lines #{17 200}}
   
    LIMITATIONS:
     - We only support gating based on line coverage. Later, we might extend this, but line coverage is a good start.
   "
  (:require [clojure.string :as s]
            [clojure.set :as clj-set]
            [taoensso.timbre :as log]))

(def ^:private named-gates 
  {:overall-coverage {:external-api-key "overall_coverage"
                      :external-name    "Overall Coverage"}
   :new-and-changed-code {:external-api-key "new_and_changed_code"
                          :external-name "New and Changed Code"}})

(def ^:private external-name->presentable-name
  (->> named-gates
       (map (fn [[_ {:keys [external-api-key external-name]}]]
              [external-api-key external-name]))
       (into {})))

(def ^:private external-name->internal-gate 
  (->> named-gates
       (map (fn [[internal-name {:keys [external-api-key]}]]
              [external-api-key internal-name]))
       (into {})))

(def ^:private all-gates-as-presentable-text
  (->> named-gates 
       vals 
       (map :external-name)
       (s/join ",")))

(defn group-consecutive-numbers [nums]
  (->> nums
       ;; (- n i) will be the same value for consecutive numbers 
       (map-indexed (fn [i n] [n (- n i)]))
       (partition-by second)
       (mapv #(mapv first %))))

(defn join-consecutive-subseries [seq-of-subseries]
  (mapv (fn [sub-series]
          (if (>= (count sub-series) 2)
            (str (first sub-series) "-" (last sub-series))
            (s/join "," sub-series)))
        seq-of-subseries))

(defn- ascending-sub-series-as-range-texts
  [a-seq-of-lines]
  (->> a-seq-of-lines
       sort
       group-consecutive-numbers
       ;; we now have series like [[1 2 3] [9] [23 24 25 26]] -> re-format to ranges if there's more than one number in a group
       join-consecutive-subseries))
(comment
  (ascending-sub-series-as-range-texts [1 2 3 4 7 11 12 13 14 18 25 26 27 31 32])
  ;; => ["1-4" "7" "11-14" "18" "25-27" "31-32"]
  )


(defn lines->presentable-text
  [a-seq-of-lines]
  (let [formatted (ascending-sub-series-as-range-texts a-seq-of-lines)
        total (count formatted)
        max-to-present 10
        base-text (->> formatted
                       (take 10)
                       (s/join ","))]
    (if (> total max-to-present)
      (str base-text "...and " (- total max-to-present) " more.")
      base-text)))

(defmulti check-criteria-for
  (fn [{:keys [coverage-gate-config] :as _gating-context}]
    (external-name->internal-gate (:name coverage-gate-config))))

(defn- create-gating-result-for
  [{:keys [coverage-gate-config] :as _gating-context}
   {:keys [pass? details percentage-covered] :as _gating-result}]
  (-> coverage-gate-config
      (assoc :pass pass?)
      (assoc :measured-coverage (or percentage-covered 0)) ; can be nil when disabled gates
      (assoc :details details)))

(defmethod check-criteria-for :default
  [{:keys [coverage-gate-config] :as gating-context}]
  (create-gating-result-for gating-context
                            {:pass? false
                             :details {:pass-or-fail-reason (str "Unknown coverage gate: " (:name coverage-gate-config)
                                                                 ". Cannot determine the pass/fail criteria. "
                                                                 "Supported gates are: " all-gates-as-presentable-text)
                                       :action "Check your coverage gates config, or contact CodeScene's support."}}))

(defn- executable-lines-with-coverage-for
  "Return a map from {line -> covered?} based on the details in the coverage report:
   
   :line-coverage {
           ...
           :details {
             :covered #{1 10 11 12 120 ...}
             :uncovered #{ 4 5 6 ...}}
           }} "
  [{:keys [lines file-name] :as _a-changed-file}
   {:keys [total details] :as _line-coverage}]
  (let [{:keys [covered uncovered]} details
        file-lacks-coverage? (every? nil? [covered uncovered])
        executable-lines (clj-set/union covered uncovered) ; TODO: this is incorrect ⚠️ A new method that isn't tested would pass the gate.
        executable-lines-changed (clj-set/intersection executable-lines lines)]
    (if file-lacks-coverage?
      (do
        (log/debug "The file " file-name " lacks coverage info -- we didn't find any coverage report for it. We have to assume that the code changes involve executable lines. "
                   (count lines) " lines were touched.")
        {:file-name file-name
         :total-loc-covered-in-file total
         :covered   []
         :uncovered lines
         :executable-lines-changed lines})
      {:file-name file-name
       :total-loc-covered-in-file total
       :covered   (->> executable-lines-changed (filter covered))
       :uncovered (->> executable-lines-changed (filter uncovered))
       :executable-lines-changed executable-lines-changed})))

(defn- coverage-of-changed-code-for
  [file->coverage
   {:keys [file-name] :as a-changed-file}]
  "The key is to evaluate the changed lines in the context of executable LoC.
   Any executable LoC that isn't covered influences the gating criteria negatively."
  (let [{:keys [details] :as my-coverage} (file->coverage file-name)
        executable-lins-coverage (executable-lines-with-coverage-for a-changed-file my-coverage)]
    (log/debug "Checking coverage for changed code. File = " file-name ", coverage exists = " (boolean (not-empty my-coverage))
               ", reported coverage = " (dissoc my-coverage :details)
               (if (seq (:uncovered executable-lins-coverage))
                 (str ". Details, uncovered LoC = " (lines->presentable-text (:uncovered executable-lins-coverage)))
                 "."))
    executable-lins-coverage))

(defn ->file->coverage
  "Creates a lookup table based on coverage info like 
    ({:path ui/preview/examples/payment-page/countries.js, 
        :name countries.js, 
        :line-coverage {
           :covered 1252, 
           :total 1252, 
       ..."
  [coverage-report]
  (->> coverage-report
       (map (fn [{:keys [path line-coverage]}]
              [path line-coverage]))
       (into {})))

(defn- sum-coverage-stat
  [coverage-status-by-file stat-to-sum]
  (->> coverage-status-by-file
       (map stat-to-sum)
       ; we started to reuse this function for overall_coverage too.
       ; the problem is that the two gates have different data types..
       (map (fn [the-stat]
              (if (seqable? the-stat)
                (count the-stat)
                the-stat)))
       (reduce +)))

(defn- round-to-one-decimal
  [float-value]
    (cond-> float-value
            (< float-value 100) (-> (* 10) float (Math/round) (/ 10.0))))

(defn- safe-percentage
  [numerator denominator]
  (if (zero? denominator)
    100
    (-> (/ numerator denominator)
        (* 100)
        round-to-one-decimal)))

(defn- percentage-covered-on-changed-code-from
  [coverage-status-by-file]
  (let [sum-total (partial sum-coverage-stat coverage-status-by-file)
        total-covered (sum-total :covered)
        total-executable (sum-total :executable-lines-changed)]
    (safe-percentage total-covered total-executable)))

(defn- percentage-covered-overall
  [coverage-status-by-file]
  (let [sum-total (partial sum-coverage-stat coverage-status-by-file)
        total-covered (sum-total :covered)
        total-executable (sum-total :total)]
    (safe-percentage total-covered total-executable)))

;; We want to give clear reasons for pass/fail. This is complicated since there 
;; are so many potential varations => use multimethods to build the diagnostics:

(defn- file-with-changed-executable-lines-lack-coverage?
  "Identify files that lack coverage, most likely because 
   they don't have any tests at all. This is something we want to 
   report as part of our failure reasons.
   
   IMPORTANT: We have to be a bit careful here. For example, an enum file will 
   have zero coverage; doesn't mean it's a problem. So ensure we also check 
   the executable lines."
  [{:keys [covered executable-lines-changed]}]
  (and (zero? covered)
       (pos? executable-lines-changed)))

(defn- coverage->explaination-context
  "Calculate the statistics needed to build a proper response to the user.
   
   Note that this context is used to both dispatch to the correct multi-method encapsulating 
   the reporting of the outcome + as part of creating the actual diagnostics/info that 
   get sent to the user."
  [coverage-status-by-file]
  (let [total-covered (sum-coverage-stat  coverage-status-by-file  :covered)
        total-uncovered (sum-coverage-stat coverage-status-by-file :uncovered)
        total-executable-changed (sum-coverage-stat coverage-status-by-file :executable-lines-changed)
        total-files (count coverage-status-by-file)
        suspects-without-coverage (filter file-with-changed-executable-lines-lack-coverage? coverage-status-by-file)]
    {:total-covered total-covered
     :total-uncovered total-uncovered 
     :total-executable-changed total-executable-changed
     :total-files total-files 
     :suspects-without-coverage suspects-without-coverage}))

(defmulti explain-outcome-of-new-and-changed-code
  (fn [{:keys [pass?]} _coverage-status-by-file _explaination-context]
    pass?))

(defmethod explain-outcome-of-new-and-changed-code false
  [{:keys [percentage-covered threshold] :as _outcome}
   coverage-status-by-file
   _explanation-context]
  (let [worst-offenders (->> coverage-status-by-file
                             (sort-by (comp count :uncovered))
                             reverse
                             (remove (comp empty? :uncovered)) ; not sure if this can happen...
                             (take 5)) ; limit the response to something reasonable
        changed-content-has-some-coverage? (->> worst-offenders
                                                (remove (comp nil? :total-loc-covered-in-file)) ; the file might lack a coverage report
                                                (remove (comp zero? :total-loc-covered-in-file))
                                                seq)
        action-files (->> worst-offenders
                          (map #(-> (assoc % :cc-details (str "uncovered line(s) = [" (lines->presentable-text (:uncovered %)) "]"))
                                    (select-keys [:file-name :cc-details]))))
        action-description (if changed-content-has-some-coverage?
                             "Add tests to: "
                             "Add tests (optionally, if they shouldn't be checked, then exclude content from the coverage gates via CodeScene's configuration) to : ")]
    {:pass-or-fail-reason (str "New or changed code lacks code coverage: " percentage-covered "% covered < threshold = " threshold "%")
     :action  {:description action-description
               :files action-files}}))

(defmethod explain-outcome-of-new-and-changed-code true ; passes the gate 👍
  [{:keys [percentage-covered threshold] :as _outcome}
   _coverage-status-by-file
   {:keys [total-files total-covered total-executable-changed]}]
  (if (zero? total-executable-changed) ; worthy of a special case to give precise info on the pass critera
    {:pass-or-fail-reason (str "No executable lines of code changed. (The change touched " total-files " files).")
     :action "-"}
    {:pass-or-fail-reason (str "New or changed code meets coverage goal: " percentage-covered "% covered >= threshold = " threshold "%")
     :action  (str "You modified " total-files " files, and covered " total-covered " added/modified lines of code.")}))

(def ^:private sensible-default-threshold 90)

; RULE: check that the total coverage -- as summed across all changed files -- exceed 
;       the threshold. Note that we do _not_ do this check per file. (That's the job of 
;       context sensitive gates, and might be too rigid for many teams).
(defmethod check-criteria-for :new-and-changed-code
  [{:keys [coverage-gate-config coverage-report changed-files] :as gating-context}]
  (let [file->coverage (->file->coverage coverage-report)
        coverage-status-by-file (map (partial coverage-of-changed-code-for file->coverage) changed-files)
        threshold (or (:threshold coverage-gate-config) sensible-default-threshold)
        percentage-covered (percentage-covered-on-changed-code-from coverage-status-by-file)
        outcome {:pass? (>= percentage-covered threshold)
                 :percentage-covered percentage-covered
                 :threshold threshold}
        ; Now we know how it went -> create a detailed response to the user:
        explanation-context (coverage->explaination-context coverage-status-by-file)
        explanation (explain-outcome-of-new-and-changed-code outcome
                                                             coverage-status-by-file
                                                             explanation-context)]
     (log/debug "New or Changed Code gate, pass? = " (:pass? outcome) " with percentage-covered = " percentage-covered " and threshold = " threshold)
    (create-gating-result-for gating-context (assoc outcome :details explanation))))

(defn- subpaths->text
  [subpaths]
  (if (empty? subpaths)
    ""
    (str " in " (s/join ", " subpaths))))

(defmulti explain-outcome-of-overall-coverage
          (fn [{:keys [pass?]} _coverage-status-by-file _explaination-context]
            pass?))

(defmethod explain-outcome-of-overall-coverage true ; passes the gate 👍
  [{:keys [percentage-covered threshold] :as _outcome}
   _coverage-status-by-file
   {:keys [total-files total-covered total-uncovered subpaths]}]
  (if (zero? total-covered)
    {:pass-or-fail-reason "No executable lines of code found."
     :action "-"}
    {:pass-or-fail-reason (format "The overall coverage gate was checked for all code%s and meets the goal: %s%% covered >= threshold = %s%%" (subpaths->text subpaths) percentage-covered threshold)
     :action (format "You have %s files with a sum of %s covered and %s uncovered lines of code." total-files total-covered total-uncovered)}))

(defmethod explain-outcome-of-overall-coverage false
  [{:keys [percentage-covered threshold] :as _outcome}
   coverage-status-by-file
   {:keys [subpaths]}]
  (let [worst-offenders (->> coverage-status-by-file
                             (sort-by :uncovered)
                             reverse
                             (remove (comp zero? :uncovered)) ; not sure if this can happen...
                             (take 5)) ; limit the response to something reasonable
        action-files (->> worst-offenders
                          (map #(-> (assoc % :cc-details (str (:uncovered %) " uncovered lines"))
                                    (select-keys [:file-name :cc-details]))))]
    {:pass-or-fail-reason (format "The overall coverage gate was checked for all code%s and is too small: %s%% covered < threshold = %s%%" (subpaths->text subpaths) percentage-covered threshold)
     :action {:description "Start by adding tests to the following files (optionally, exclude content from the coverage gates via CodeScene's configuration): "
              :files action-files}}))

(defmethod check-criteria-for :overall-coverage
  [{:keys [coverage-gate-config coverage-report] :as gating-context}]
  (let [subpaths (->> coverage-report (map :subpath) (filter not-empty) set)
        coverage-status-by-file (map #(-> (:line-coverage %)
                                          (assoc :file-name (:path %)
                                                 :executable-lines-changed 0 ;do not know yet how to compute it
                                                 :uncovered (- (get-in % [:line-coverage :total] 0)
                                                               (get-in % [:line-coverage :covered] 0))))
                                     coverage-report)
        threshold (or (:threshold coverage-gate-config) sensible-default-threshold)
        percentage-covered (percentage-covered-overall coverage-status-by-file)
        outcome {:pass?              (>= percentage-covered threshold)
                 :percentage-covered percentage-covered
                 :threshold          threshold}
        ; Now we know how it went -> create a detailed response to the user:
        explanation-context (coverage->explaination-context coverage-status-by-file)
        explanation (explain-outcome-of-overall-coverage outcome
                                                         coverage-status-by-file
                                                         (assoc explanation-context :subpaths subpaths))]
    (log/debug "Overall Coverage gate, pass? = " (:pass? outcome) " with percentage-covered = " percentage-covered " and threshold = " threshold)
    (create-gating-result-for gating-context (assoc outcome :details explanation))))

(defn- explain-disabled-gate
  [coverage-gate-config]
  {:pass-or-fail-reason (str "Disabled gate: The " (external-name->presentable-name (:name coverage-gate-config)) " gate is disabled.")
   :action "-"})

(def ^:private gate-enabled? :enabled)

(defn- check-when-gate-enabled
  [{:keys [coverage-gate-config] :as gating-context}]
  (log/debugf "Checking the gate: %s. Gate enabled? = %s" (:name coverage-gate-config) (:enabled coverage-gate-config))
  (if (gate-enabled? coverage-gate-config)
    (check-criteria-for gating-context)
    (create-gating-result-for gating-context {:pass? true
                                              :details (explain-disabled-gate coverage-gate-config)})))

(defn check-all-gates
  [{:keys [coverage-gates-config ; as specified in https://codescene.slab.com/posts/code-coverage-quality-gates-cc-gq-in-pull-requests-pr-68mr7ciy
           _coverage-report       ; our internal format :details and :uncovered are optional
                                  ; [{:path "some/path/Test.java" :name "Test.java" :line-coverage {:covered  2 :total 4 :coverage 1/2 :details {:covered #{1 2}, :uncovered #{3 4}}}}]
           _changed-files]        ; a git diff telling us what happened on a feature branch, or empty in case of main/trunk-based
    :as gating-context-for-all-gates}]
  (->> coverage-gates-config
       (map (fn [a-gate]
              (check-when-gate-enabled (assoc gating-context-for-all-gates :coverage-gate-config a-gate))))))

(defn gates-require-full-coverage-report?
  "Some gates require a coverage report over the whole codebase, not only the changed files.
   That operation is quite costly in terms of run-time: we need to estimate executable and 
   uncovered LoC across -- potentially -- thousands of files.
   This function checks whether that full information is needed. If not, then the CLI can skip 
   that step in the data mining and save a lot of time for the check run."
  [coverage-gates-config]
  (let [requires-full-coverage? #{:overall-coverage}
        enabled-gates (filter gate-enabled? coverage-gates-config)]
    (some (comp requires-full-coverage? external-name->internal-gate :name) enabled-gates)))
