(ns codescene.features.api.routes.code-coverage
  (:require [clojure.spec.alpha :as s]
            [codescene.features.api.code-coverage :as code-coverage]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.spec.code-coverage :as api-code-coverage-spec]
            [codescene.features.api.spec.common :as common-spec]
            [codescene.features.components.auth :as auth]
            [codescene.features.components.ff :as ff]
            [codescene.features.util.api :as api-utils]
            [codescene.features.util.maps :as maps]
            [compojure.api.sweet :refer [GET POST PUT DELETE context undocumented]]
            [spec-tools.data-spec :as ds]
            [spec-tools.spec :as spec]))

(def ^:const outcomes-page-limit 100)
(s/def ::date_type (s/and string? #(not (empty? %)) #(re-matches common-spec/YYYY-MM-DD %)))

(defn- user
  [system request]
  (auth/user (api-core/api-auth system) request))

(defn- code-coverage-read-routes
  [system]
  (context "/" []
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read api-privileges/cli-access})]
    (GET "/" req
      :summary "Get metadata for uploaded code coverage data"
      :tags ["code-coverage"]
      :query-params [{repo-url :- ::api-code-coverage-spec/repo-url ""}]
      :responses {200 {:schema      spec/any?
                       :description "metadata for uploaded code coverage data"}}
      (code-coverage/get-data system (user system req) {:repo-url repo-url}))))

(defn- code-coverage-write-routes
  [system]
  (context "/" []
    ;;we need to give access to restapi users and to cli users
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-write api-privileges/cli-access})]
    (undocumented
      (POST "/upload" req
        :summary "Request uploading code coverage data"
        :tags ["code-coverage"]
        :body [data (ds/spec {:name :metadata
                              :spec (s/keys :req-un [::api-code-coverage-spec/repo-url
                                                     ::api-code-coverage-spec/commit-sha
                                                     ::api-code-coverage-spec/format
                                                     ::api-code-coverage-spec/metric]
                                            :opt-un [::api-code-coverage-spec/repo-path
                                                     ::api-code-coverage-spec/subpath])})]
        :responses {200 {:schema      ::api-code-coverage-spec/upload-id
                         :description "an id to use when uploading"}}
        (code-coverage/request-upload system (user system req) data)))
    (undocumented
      (PUT "/upload/:upload-id" req
        :summary "Upload code coverage data"
        :tags ["code-coverage"]
        :responses {201 {:schema      spec/any?
                         :description "upload code coverage data"}}
        :path-params [upload-id :- spec/pos-int?]
        (code-coverage/upload-data system (user system req) upload-id (:body req))))
    (DELETE "/" req
      :summary "Delete code coverage data"
      :tags ["code-coverage"]
      :body [query-params (ds/spec {:name :query-params
                                    :spec (s/keys :req-un [::api-code-coverage-spec/repo-url])})]
      :responses {201 {:schema      spec/any?
                       :description "delete code coverage data"}}
      (code-coverage/delete-data system (user system req) query-params))))

(defn check-routes
  [system project-id]
  (context "/" []
    (GET "/config" req
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read api-privileges/cli-access})]
      :summary "Get project code coverage config"
      :tags ["code-coverage"]
      :responses {200 {:schema      spec/any?
                       :description "project code coverage config"}}
      (api-utils/ok (code-coverage/get-project-config system project-id)))
    (POST "/result" req
      :summary "Posting the code coverage check results"
      :tags ["code-coverage"]
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write api-privileges/cli-access})]
      :body [data (ds/spec {:name ::check-results
                            :spec {:coverage_results spec/any?
                                   :commit_sha ::api-code-coverage-spec/commit-sha
                                   :repo_url ::api-code-coverage-spec/repo-url
                                   (ds/opt :base_ref) (ds/maybe spec/string?)}})]
      :responses {200 {:schema spec/any?
                       :description "code coverage check post result"}}
      (code-coverage/check-result system (user system req) project-id (maps/->kebab data)))))

(defn gate-results-routes
  [system project-id]
  (context "/gate-results" []
    (GET "/insights" req
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read api-privileges/cli-access})]
      :summary "Get project code coverage insights results"
      :tags ["code-coverage"]
      :query-params [{from :- ::date_type ""} {to :- ::date_type ""}]
      :responses {200 {:schema      spec/any?
                       :description "project code coverage insights results"}}
      (code-coverage/insights system (maps/map-of project-id from to)))
    (GET "/outcomes" req
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read api-privileges/cli-access})]
      :summary "Get project code coverage outcomes results"
      :tags ["code-coverage"]
      :query-params [{from :- ::date_type ""} {to :- ::date_type ""}
                     {page :- spec/pos-int? 1}
                     {page-size :- spec/pos-int? outcomes-page-limit}]
      :responses {200 {:schema      spec/any?
                       :description "project code coverage outcomes results"}}
      (code-coverage/outcomes system (maps/map-of project-id from to page page-size)))))

(defn sub-routes
  [system]
  (context "/code-coverage" []
    :middleware [#(api-core/wrap-can-use-feature system % :code-coverage?)]
    (code-coverage-read-routes system)
    (code-coverage-write-routes system)
    (context "/projects/:project-id" [project-id]
      :path-params [project-id :- spec/pos-int?]
      (undocumented (check-routes system project-id))
      (when (ff/on? (api-core/api-ff system) :code-coverage-gates-rest-api)
        (gate-results-routes system project-id)))))

(comment
  (clj-http.client/get
    "http://localhost:4000/v2/code-coverage/projects/2/config"
    {:as :json
     :oauth-token "pat_eyJpZCI6MSwicmFuZCI6Im1waUx0V1hIMXZySGF4b1piL2kwNUhHUGZtRVRuUnJkIn0"}))
