(ns codescene.features.analytics.handlers
  (:require
   [clojure.string :as str]
   [codescene.features.util.url :as url]
   [codescene.util.http :as http]
   [medley.core :as m]))


;; allow only specific event types because this is an internal API
;; and we do not want to send garbage to event stores;
;; e.g. pentest scans tend to generate non-sensical event types (https://empear.slack.com/archives/C01U78V87JM/p1674642790836109)
(defn- valid-event-type? [event-type]
  (and (not (str/blank? event-type))
       (#{"feature-access" "userpilot-flow"} event-type)))

(defn- valid-event-source? [event-source]
  (or (not event-source)
      (str/blank? event-source)
      (#{"codescene-ui" "userpilot"} event-source)))

(defn- valid-uri? [uri]
  (try
    (boolean (url/to-uri uri))
    (catch java.net.URISyntaxException _syn
      false)))

(def ^:private messages {:event-type "Invalid event-type."
                         :event-source "Invalid event-source."
                         :uri "Invalid or malformed uri."})

(defn- validate
  [{:keys [event-type event-source uri]}]
  (->> [{:pred valid-event-type? :val event-type :key :event-type}
        {:pred valid-event-source? :val event-source :key :event-source}
        {:pred valid-uri? :val uri :key :uri}]
       (remove (fn [{:keys [pred val]}] (pred val)))
       first))


(defn ui-events
  "Handle events generated by client, e.g. Userpilot events or UI events from codescene-ui.

   `send-event-fn` and `feature-menu-item-fn` must be the
  product-specific version of those functions. 

  `send-event-fn` is responsable for actually doing something with the
  event. This function must take two arguments: the request (required
  in Cloud) and the event to be sent.

  `feature-menu-item` takes a URI as an argument and returns the
  corresponding menu keyword. See
  `codescene.features.menus/match-path-to-menu-item`.

  This handler would typically be called
  like this:
  
  ```
  (POST /my-path req (ui-events req send-event feature-menu-item)) 
```

  `key-separator` is necessary because the top-level keys are
  formatted differently for Cloud and Onprem: Cloud uses _ like
  Amplitude, but Onprem uses the more civilized - because it's talking
  to a service written in Clojure.
 "
  ([req send-event-fn  feature-menu-item-fn]
   (ui-events req send-event-fn feature-menu-item-fn "_"))
  ([{:keys [params] :as req} send-event-fn  feature-menu-item-fn key-separator]
   (let [{:keys [event-source event-type uri]} params
         event-type-key (keyword (str "event" key-separator "type"))
         event-properties-key (keyword (str "event" key-separator "properties"))]
     (if-let [invalid-param (validate params)]
       (http/response {:status :error :message (get messages (:key invalid-param))} 400)
       (do
         (send-event-fn req {event-type-key (keyword event-type)
                             event-properties-key (-> params
                                                      (dissoc :event-type)
                                                      (m/assoc-some
                                                       :event-source event-source
                                                       :feature-menu-item (when (str/starts-with? event-type "feature-access")
                                                                            (feature-menu-item-fn uri))))})
         (http/response {:status :ok}))))))
