(ns codescene.features.analysis.files
  (:require [codescene.analysis.paths :as paths]
            [codescene.analysis.architecture.aggregator :as aggregator]
            [codescene.analysis.analysis-context-builder :as analysis-context-builder]
            [codescene.biomarkers.architectural-level-hotspots :as architectural-hotspots]
            [codescene.code-health.persisted-code-health-details :as persisted-code-health-details]
            [codescene.code-health.queries :as code-health]
            [codescene.util.json :as json]
            [codescene.features.util.number :as number-util]
            [codescene.note-watch.supervised-scores :as supervised-scores]
            [codescene.presentation.display :as display]
            [hotspots-x-ray.recommendations.code-health-interpretations :as chi]
            [clojure.core.memoize :as memo]
            [clojure.set :as set]
            [clojure.string :as string]
            [clojure.java.io :as io]
            [semantic-csv.core :as sc]
            [medley.core :as m]
            [spec-tools.parse]))

(def ^:const bio-marker-severity [{:indication 3 :severity "alert"}
                                  {:indication 2 :severity "warning"}
                                  {:indication 1 :severity "improvement"}])

(def ^:const indication->rule-set {3 "critical"
                                   2 "advisory"})

(defn- numeric-code-health
  [v]
  (and v
       (display/->maybe-double v)))

(defn- lookupable-code-health-scores-depending-on-analysis
  "If we have the full scan data, then we'll use that. Otherwise we
   fall back on the older API and only present hotspots code health."
  [analysis-path-fn]
  (if (.exists (io/file (analysis-path-fn paths/full-scan-code-health-csv)))
    (->> (analysis-path-fn paths/full-scan-code-health-csv)
         sc/slurp-csv
         (map (fn [{:keys [name now month year]}]
                [name {:now (numeric-code-health now)
                       :last-month (numeric-code-health month)
                       :last-year (numeric-code-health year)}]))
         (into {}))
    (code-health/lookupable-file-level-hotspots analysis-path-fn)))

(defn- process-social-individual-data
  [data]
  (map #(-> %
            (select-keys [:ownership :path :nauthors :fragmentation])
            (assoc :path (string/join "/" (:path %)))
            (set/rename-keys {:ownership :main_author_contribution_percentage
                              :nauthors :number_of_authors
                              :fragmentation :author_contributions_fragmentation})) data))

(defn- process-social-team-data
  [data]
  (map #(-> %
            (select-keys [:owner :ownership :path :nauthors :fragmentation])
            (assoc :path (string/join "/" (:path %)))
            (set/rename-keys {:owner :primary_team
                              :ownership :team_ownership
                              :nauthors :number_of_teams
                              :fragmentation :teams_contributions_fragmentation})) data))

(defn- flatten-tree
  "flat recursively all :children from the given data"
  [data]
  (if-let [children (:children data)]
    (flatten (map flatten-tree children))
    data))

(defn- get-social-team-system-map
  [analysis-dir process-fn file]
  (let [json-data (json/read-when-exists analysis-dir file)]
    (when (:children json-data)
      (-> json-data
          flatten-tree
          process-fn))))

(defn- social-data-for
  [analysis-dir]
  (let [individual-social-data (get-social-team-system-map analysis-dir process-social-individual-data paths/social-system-map-file-name)
        social-team-system-map (get-social-team-system-map analysis-dir process-social-team-data paths/social-team-system-map-file-name)
        file->team-data (group-by :path social-team-system-map)]
    (->> individual-social-data
         (map (fn [{:keys [path] :as individual}]
                (merge individual (first (get file->team-data path))))))))

(defn- rename-map-keys
  [map]
  (set/rename-keys map {:ndefects      :number_of_defects
                        :revs          :change_frequency
                        :ownership     :ownership_percentage
                        :defectdensity :defectdensity_percentage
                        :loss          :knowledge_loss_percentage
                        :size          :lines_of_code
                        :age           :last_modification_age_in_months}))

(defn get-recommendations-for-markers
  [context markers]
  (let [interpretation (if (true? (:calculated markers)) (chi/interpret-for-a-human context markers) [])]
    (map (fn [item]
           (-> (merge item (first (filter #(= (:indication %) (:indication item)) bio-marker-severity)))
               (select-keys [:title :severity])))
         interpretation)))

;; Cached version of `persisted-code-health-details/code-health-details-for` to avoid rereading the same code health files
;; on consecutive api calls (when fetching pages of data with a script)
(def ^:private code-health-details-for (memo/ttl persisted-code-health-details/code-health-details-for :ttl/threshold 60000))

(defn get-rule-violations-for-markers
  [context markers]
  (let [interpretation (if (true? (:calculated markers)) (chi/interpret-for-a-human context markers) [])]
    (->> interpretation
         (keep (fn [{:keys [title indication functions] :as _item}]
                 (when-let [rule-set (indication->rule-set indication)]
                   (m/assoc-some {:code_smell title
                                  :rule_set rule-set}
                                 :count (when (seq functions) (count functions))))))
         (into []))))

(defn- get-goals-for
  [_context notes name]
  (->> notes
       (filter #(string/includes? (:name %) name))
       (map :note)
       (map (fn [{:keys [category note-text]}]
              {:category category
               :text note-text}))))

(defn- format-score [x]
  (if (and (number? x) (pos? x))
    (number-util/round-code-health-scores x)
    ;; TODO: knorrest - why return a string representation for "no score"?
    "-"))

(defn- code-health-for-path
  [lookupable-scores path]
  (let [{:keys [now last-month last-year]} (get lookupable-scores path)]
    {:year_score  (format-score last-year)
     :month_score (format-score last-month)
     :current_score (format-score now)}))

(defn- process-file-data
  [path->additional-data social-team-system-map data]
  (let [data-with-social (set/join (set (map #(assoc % :path (string/join "/" (:path %))) data)) (set social-team-system-map))]
    (map (fn [{:keys [path] :as item}]
           (-> (merge {:number_of_defects 0 :cost 0} item)
               rename-map-keys
               (select-keys [:path :lines_of_code :last_modification_age_in_months :name :language
                             :ownership_percentage :owner :cost :knowledge_loss_percentage
                             :change_frequency :system_health :number_of_defects :team_ownership :primary_team
                             :number_of_teams :number_of_authors :author_contributions_fragmentation
                             :teams_contributions_fragmentation])
               (m/update-existing :ownership_percentage number-util/floor-number-when-exists)
               (m/update-existing :knowledge_loss_percentage number-util/floor-number-when-exists)
               (m/update-existing :team_ownership number-util/floor-number-when-exists)
               (m/update-existing :number_of_teams display/->maybe-int)
               (m/update-existing :number_of_authors display/->maybe-int)
               (m/update-existing :author_contributions_fragmentation number-util/floor-number-when-exists)
               (m/update-existing :teams_contributions_fragmentation number-util/floor-number-when-exists)
               (merge (path->additional-data path))))
         data-with-social)))

(defn- hotspots [{:keys [dir path-fn] :as _analysis} component]
  (let [score-path (if (= :system component)
                     (path-fn paths/code-bio-marker-scores-csv)
                     (architectural-hotspots/score-path dir component))]
        (->> (sc/slurp-csv score-path)
             (filter #(= "1" (:candidatescore %)))
             (mapv :name))))

(defn- ->hotspot-p [analysis component]
  (let [hotspots (set (hotspots analysis component))]
    (fn [path]
      (contains? hotspots path))))

(defn- transformations-from
  [analysis-path-fn]
  (-> paths/architectural-transformations-csv analysis-path-fn sc/slurp-csv))

(defn- ->component-p
  [{:keys [path-fn] :as _analysis} component]
  (if (= :system component)
    (constantly true)
    (let [transformations (transformations-from path-fn)
          path->component (aggregator/make-file->component-mapper transformations)]
      (fn [{:keys [path]}]
        (let [path (string/join "/" path) ]
          (= component (path->component path)))))))

(defn file-list
  "return the analysis file list with code health and known biomarker "
  ([analysis]
   (file-list analysis :system))
  ([analysis component]
   (let [analysis-dir (:dir analysis)
         context (assoc (analysis-context-builder/build-context-for-existing-analysis (partial paths/make-analysis-path-to analysis-dir))
                        :analysis-path analysis-dir)
         lookupable-scores (lookupable-code-health-scores-depending-on-analysis (partial paths/make-analysis-path-to analysis-dir))
         social-team-system-map (social-data-for analysis-dir)
         files-data (json/read-when-exists analysis-dir paths/system-map-json)
         notes (->> (json/read-when-exists analysis-dir paths/risks-for-notes-json)
                    (map :note-score)
                    (map :note)
                    (map supervised-scores/file-details-for-note))
         hotspot? (->hotspot-p analysis component)
         path->additional-data (fn [path]
                                ;; Note that here we read a separate details json file for every file in the analysis
                                 (let [markers (code-health-details-for analysis-dir path)]
                                   {:goals (get-goals-for context notes path)
                                    :code_health (code-health-for-path lookupable-scores path)
                                    :code_health_rule_violations (get-rule-violations-for-markers context markers)
                                    :hotspot (hotspot? path)
                                   ;; TODO: Deprecate (=remove) the recommendations property (in favour of code_smells) 
                                    :recommendations (get-recommendations-for-markers context markers)}))]
     (when (:children files-data)
       (->> files-data
            flatten-tree
            (filter (->component-p analysis component))
            (process-file-data path->additional-data social-team-system-map))))))
