// @flow
import $ from 'jquery';
import * as xhr from 'codescene-ui/src/browser/xhr';
import * as modernizedTable from 'codescene-ui/src/shared/modernized-table/modernized-table';

import * as modal from 'codescene-ui/src/modal.js';
import * as alerts from 'codescene-ui/src/alerts';
import * as viewer from 'codescene-ui/src/analyses/view-code/view-code';
import * as codeBlock from 'codescene-ui/src/shared/code-tools/code-block';

const titleMappings = {
  'Lines added': 'LOCAdded',
  'Lines deleted': 'LOCDeleted',
};

const reversedTitleMappings = {
  LOCAdded: 'Lines added',
  LOCDeleted: 'Lines deleted',
};

function fetchFunctionContents(url) {
  return xhr.get(url);
}

function showFunctionCode(url, title) {
  const container = $('<div>').addClass('function-code');
  const progressBar = alerts.progress(container, 'Loading...');

  modal.show({
    wrapperClass: 'function-code-modal',
    title,
    content: container,
    size: 'lg'
  });

  fetchFunctionContents(url)
    .then((r) => {
      progressBar.remove();
      // on-prem returns raw text, cloud returns json, so we don't
      // know the content-type until we get the response (and can't
      // use our usual json fetching function).
      const contentType = r.getResponseHeader('Content-Type');
      const isJson = contentType && contentType.match(/^application\/json.*/);
      if (isJson) {
        const func = JSON.parse(r.responseText)[0];
        viewer.showCodeForFunction(
          container,
          null,
          { includeSourceInfo: true, name: null, startLine: func['start-line'], endLine: func['end-line'], ...func }
        );
      }
      else {
        codeBlock.init(container[0], {
          code: r.responseText,
        });
      }
    }).catch(err => {
      console.error(err.stack);
      alerts.error(container, 'Failed to show function code.');
    });
}

type ViewOptions = {
  projectId: number,
  analysisId: number,
  commitsUrl: string,
  pmIssuesUrl: string,
  commitsToTicketIdsUrl: string;
};

function parseCommit(row, commitsToTicketIds) {
  return {
    commit: row.rev,
    date: row.date,
    files: row.files,
    LOCAdded: row.addedcode,
    LOCDeleted: row.deletedcode,
    issue: commitsToTicketIds[row.rev],
    repository: row.repository
  };
}

function parseIssue(row) {
  return {
    id: row.id,
    href: row.href
  };
}

function renderTable(container, commits, issues, projectId, analysisId) {

  const repoNameByCommit = commits.reduce((m, x) => {
    m[x.commit] = x.repository;
    return m;
  }, {});

  const hrefByIssue = issues.reduce((m, x) => {
    m[x.id] = x.href;
    return m;
  }, {});

  const titles = Object.keys(commits[0]).map((item) => ({
    key: reversedTitleMappings[item] || item,
    value: titleMappings[item] || item,
  }));

  const list = commits.map((item) => {
    const obj = { ...item };
    const repoName: string = encodeURIComponent(repoNameByCommit[item.commit]);
    const commit: string = encodeURIComponent(item.commit);

    const commitUrl = `${window.baseUrl}${projectId}/analyses/${analysisId}/scope/analysis-data/commits/details.txt?repo=${repoName}&commit=${commit}`;
    obj.commit = {
      value: item.commit,
      title: `View commit ${item.commit} details.`,
      onClick: () => showFunctionCode(commitUrl, `Commit ${item.commit}`)
    };

    obj.files = {
      value: item.files,
      sortCallback: () => item.files && parseInt(item.files, 10),
    };

    obj[titleMappings['Lines added']] = {
      value: item.LOCAdded,
      sortCallback: () => item.LOCAdded && parseInt(item.LOCAdded, 10),
    };

    obj[titleMappings['Lines deleted']] = {
      value: item.LOCDeleted,
      sortCallback: () => item.LOCDeleted && parseInt(item.LOCDeleted, 10),
    };

    if (item.issue) {
      const href = hrefByIssue[item.issue];
      if (href) {
        obj.issue = {
          value: item.issue,
          title: `View issue ${item.issue} in external system.`,
          href
        };
      }
    }

    return obj;
  });

  modernizedTable.init({
    mountPoint: container[0],
    usePager: true,
    titles: titles,
    list,
    useInPaperCardWithNegativeMargins: true
  });
}

export function init(options: ViewOptions) {
  const container = $('#tableview');

  xhr.json(options.commitsToTicketIdsUrl)
    .then(commitsToTicketIds => {
      xhr.csv(options.pmIssuesUrl, parseIssue)
        .then(issues => {
          xhr.csv(options.commitsUrl, row => parseCommit(row, commitsToTicketIds))
            .then(commits => {
              renderTable(container, commits, issues, options.projectId, options.analysisId);
            });
        })
        .catch(() => {
          // No pm-issues available -> go on without them
          xhr.csv(options.commitsUrl, row => parseCommit(row, commitsToTicketIds))
            .then(commits => {
              renderTable(container, commits, [], options.projectId, options.analysisId);
            });
        });
    });
}
