grammar GoNestedComplexity;

import GoSharedRulesLexer;

// Override the general Lexer rule since Go supports data declarations inside curly braces, and these
// declarations might be part of conditionals. For example:
//
// if err := daemon.ContainerRm(cid, &types.ContainerRmConfig{ForceRemove: true, RemoveVolume: true}); err != nil {
//    logrus.Errorf("Failed to remove container %s: %s", cid, err)
// }
//
// The combination of LeftBrace NEWLINE uniquely identifies the start of the conditional/loop body.
NEWLINE : '\r'? '\n';

// This grammar detects islands of nested complexity.
// Note that in Go, curly braces are mandatory around loops and conditionals.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_clause
          | switch_clause
          | for_loop;

block_expression: LeftBrace expression*? RightBrace;

anything: ~(LeftBrace | RightBrace);

if_clause: IF some_condition multi_line_conditional_expression;
else_clause: ELSE multi_line_conditional_expression;

switch_clause: SWITCH some_condition multi_line_conditional_expression;

multi_line_conditional_expression: LeftBrace NEWLINE expression*? RightBrace;

for_loop: FOR for_loop_condition multi_line_conditional_expression;
for_loop_condition: ~LeftBrace*?; // optional part

some_condition: (conditional_operator | .)+?;

conditional_operator: AND | OR;

