(ns codescene.features.util.retry
  (:require [taoensso.timbre :as log]))

(defn exponential-backoff
  "This will retry after waiting 'time' ms the first time, then '* rate' it
  each time. When it waits 'max' ms, it won't retry any more. Predicate 'p?'
  applied on the Throwable decides whether to retry or fail."
  [{:keys [time rate max p?]
    :or {time 250 rate 2 max 2000 p? (constantly true)}
    :as opts} f]
  (try
    (f)
    (catch Throwable t
      (cond
        (>= time max) (do (log/warnf "exponential-backoff: max time reached will not retry" t)
                          (throw t))
        (p? t) (do
                 (log/warnf t "exponential-backoff: caught exception. Will retry in %s ms" time)
                 (Thread/sleep (long time))
                 (exponential-backoff (assoc opts :time (* time rate)) f))
        :else (do
                ;; log only exception message, not full stacktrace
                ;; (annoying when we don't want to retry (such as 404 on S3 download))
                (log/warn "exponential-backoff: non-handled exception, will not retry." t)
                (throw t))))))

(defmacro try-backoff
  "See `exponential-backoff` for options description.
  Use like:
   (try-backoff {}
     (println \"trying!\")
     (do-some-stuff))"
  [opts & body]
  `(exponential-backoff ~opts (fn [] ~@body)))

(defn try-times*
  "This will try f n times if failing. Predicate 'p?' applied on the Throwable decides whether to retry or fail."
  [{:keys [n p?] :as opts} f]
  (try
    (f)
    (catch Throwable t
      (cond
        (>= 1 n) (throw t)
        (p? t) (try-times* (assoc opts :n (dec n)) f)
        :else (throw t)))))

(defmacro try-times
  "Use like:
     (try-times {:n 2 :p? some-predicate}
       (println \"trying!\")
       (do-some-stuff))"
  [opts & body]
  `(try-times* ~opts (fn [] ~@body)))
