(ns codescene.features.repository-provider.github.app
  "Common authentication & installation functionality for GitHub Apps.
  See `codescene.features.repository-provider.github.checks` and Delta analysis for some use cases.

  To get anything done with GitHub Apps you need to first generate a JWT token (max validity 10 mins)
  and then exchange it for the app _installation token_ (max validity 1 hour).
  To get an installation token you need the JWT and an installation id - the id is either received
  in a GitHub webhook event, or you can find matching installation if you have a repository URL
  where the app has already been installed."
  (:require [buddy.core.keys :as buddy-keys]
            [buddy.sign.jwt :as buddy-jwt]
            [codescene.cache.core :as cache]
            [codescene.features.client.api :refer [to-url]]
            [codescene.features.repository-provider.github.api :as api]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(defn- jwt-data [app-id time-now-seconds max-age-seconds]
  {"iat" time-now-seconds
   "exp" (+ time-now-seconds max-age-seconds)
   "iss" app-id})

;; https://funcool.github.io/buddy-sign/latest/
(defn- generate-jwt
  "Generates a new temporary jwt token for given GitHub app which can be then
  exchanged for an installation token.

  The maximum max-age-seconds allowed by Github is 10 minutes, but it's been shown to
  cause issues (rarely and sporadically). The default value is 8 minutes instead."
  ([app-id app-private-key]
   (generate-jwt app-id app-private-key (* 8 60)))
  ([app-id app-private-key max-age-seconds]
   (buddy-jwt/sign (jwt-data app-id
                             (quot (System/currentTimeMillis) 1000)
                             max-age-seconds)
                   ;; key must be an instance of java.security.PrivateKey
                   (buddy-keys/str->private-key app-private-key)
                   {:alg :rs256})))

(defn get-user-installation
  "Given github username returns the GitHub app installation details or nil if it's not installed.
  The username can be a real user or an organization - basically any 'owner-login'.

  Note that it doesn't matter whether you choose \"All repositories\" or \"Only select repositories\";
  the user installation still returns similar data and you have to list repositories
  to check which of them are installed.

  See https://docs.github.com/en/free-pro-team@latest/rest/reference/apps#get-a-user-installation-for-the-authenticated-app"
  [jwt-authed-client github-username]
  (try+
    (api/api-request jwt-authed-client :get (to-url "/users" github-username "installation") {})
    (catch [:type :http-error :status 403] _
      ;; no installation found
      )))

(defn- get-installation-token
  "Once we have installation id (:app_id) we can get 'installation access token'
  to be able to call the Check Run API:   Installation tokens are valid for one hour.
  Returns nil if app-installation-id is empty.
  See https://developer.github.com/v3/apps/#create-a-new-installation-token."
  [jwt-authed-client app-installation-id ->authed-client]
  ;; :permissions {:checks write, :metadata read, :pull_requests write}
  ;;              {:checks write, :contents read, :metadata read, :pull_requests write}
  (let [url (to-url "/app/installations" app-installation-id "access_tokens")
        {:keys [token permissions]} (api/api-request jwt-authed-client :post url {})]
    (log/infof "Created GitHub App installation %s token with permissions %s" app-installation-id permissions)
    ;; Here we're counting on ->authed-client being IObj, let's save those permissions on the token
    (some-> token ->authed-client (with-meta {:permissions permissions}))))

(defn get-installation-repositories
  "List repositories that the app installation can access.
  This is very handy when you need to get all repositories for given user/owner.
  Use `get-user-installation-token` to get the installation token.

  Returns full details as per the `:repositories` key in the response.
  Returns empty collection if no installation token is provided.
  Uses paging (100 items) to get all the repositories.
  See https://docs.github.com/en/free-pro-team@latest/rest/reference/apps#list-repositories-accessible-to-the-app-installation"
  [installation-token-authed-client]
  (log/info "Fetch all installation repositories.")
  (api/get-all-pages installation-token-authed-client
                     "/installation/repositories"
                     {:items [:body :repositories]}))

(defn- -installation-token-authed-client
  "Creates an authed client whose method of authentication is a token for the specified
  app for the specified installation ID (or a login can be provided)"
  {cache/conf (cache/ttl 60 (fn [_ app-id _ id] [app-id id]))}
  [->authed-client app-id private-key login-or-installation-id]
  (let [jwt-authed-client (->authed-client (generate-jwt app-id private-key))
        installation-id (condp instance? login-or-installation-id
                          String (:id (get-user-installation jwt-authed-client login-or-installation-id))
                          Number login-or-installation-id)]
    (when installation-id
      (get-installation-token jwt-authed-client installation-id ->authed-client))))

(cache/memo #'-installation-token-authed-client)

(defn installation-token-authed-client
  "Creates an authed client whose method of authentication is a token for the specified
  app for the specified installation ID (or a login can be provided)"
  [->authed-client app-id private-key login-or-installation-id]
  (when (and app-id private-key)
    (if login-or-installation-id
      (-installation-token-authed-client ->authed-client app-id private-key login-or-installation-id)
      (->authed-client (generate-jwt app-id private-key)))))