
(ns codescene.features.pm-data.jira.jira-api
  "Contains methods for fetching data over the Jira Rest API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require [clj-http.client :as client]
            [taoensso.timbre :as log]
            [clojure.string :as string]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.http-helpers :as h]
            [codescene.features.util.log :as u.log]
            [codescene.features.util.retry :as retry]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]))

(def ^:const ^:private jira-cloud-rest-api-path
  "/rest/api/3")

(def ^:const ^:private jira-server-rest-api-path
  "/rest/api/2")

(defn is-jira-cloud? [api-url]
  (string/includes? api-url "atlassian.net"))

(defn api-subpath [api-url]
  (if (is-jira-cloud? api-url)
    jira-cloud-rest-api-path
    jira-server-rest-api-path))

(defn append-api-subpath
  [params] 
  (update params :api-url #(str % (api-subpath %))))

(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :content-type :json
                  :accept       :json
                  :as           :json-strict
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         append-api-subpath ;; Note that this must be after augment-request, but before fix-api-url
                         (api-client/fix-api-url nil))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch Jira data from " (:url final-params))
      (client/request final-params))))

(defn api-request
  [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(def ^:const default-http-page-size 100)
(def ^:const retry-time-ms 20000)
(def ^:const retry-max-time-ms 30000)

(defn- retryable-http-error
  [e]
  (let [{:keys [type status]} (ex-data e)]
    (and (= :http-error type)
         ;; Any http response that has a http status, 
         ;; we consider retryable. If it has no status, 
         ;; then it's not a valid HTTP response, and something
         ;; went wrong somewhere, so we throw an exception.
         status)))

(defn get-data [api-client url query-params]
  (api-client/check-interrupt)
  (api-request api-client :get url
               {:query-params   query-params}))

(defn post-data [api-client url form-params]
  (api-request api-client :post url
               {:form-params   form-params}))

(defn- get-data-with-retry
  [api-client url query-params]
  (retry/try-backoff
    {:time retry-time-ms :rate 1.1 :max retry-max-time-ms :p? retryable-http-error}
    (get-data api-client url query-params)))

(defn get-data-fn [retry?]
  (if retry? get-data-with-retry get-data))

(defn- get-paged-data 
  [api-client 
   url 
   {:keys [data-field query-params page-size retry? transformer]
    :or {query-params {} page-size default-http-page-size transformer identity} :as _options}]
  (loop [start-at 0
         max-results (or page-size default-http-page-size)
         total-count nil
         acc-data []]
    (log/infof "Fetching paged jira data from %s starting at %d%s..."
               url start-at (if total-count (format " (of %s)" total-count) ""))
    (let [query-params (merge query-params
                              {:startAt start-at
                               :maxResults max-results})
          result ((get-data-fn retry?) api-client url query-params)
          {max-results-received :maxResults data data-field :keys [total isLast]} result]
      (if (or isLast (<= total (+ (count data) (count acc-data))))
        (into acc-data (map transformer) data)
        (recur (+ max-results-received start-at) max-results-received total (into acc-data (map transformer) data))))))

(defn- get-paged-data-new
  "Implements the new paging mechanism using nextPageToken that is used
   for the new jql search endpoins"
  [api-client
   url
   {:keys [data-field query-params page-size retry? transformer]
    :or {query-params {} page-size default-http-page-size transformer identity} :as _options}]
  (loop [next-page-token nil
         max-results (or page-size default-http-page-size)
         acc-data []]
    (log/infof "Fetching paged jira data from %s, got %s items so far..." url (count acc-data))
    (let [query-params (merge query-params
                              {:nextPageToken next-page-token
                               :maxResults max-results})
          result ((get-data-fn retry?) api-client url query-params)
          {max-results-received :maxResults data data-field :keys [nextPageToken isLast]} result]
      (if isLast
        (into acc-data (map transformer) data)
        (recur nextPageToken max-results-received (into acc-data (map transformer) data))))))

(defn fetch-project
  "Fetches project from the remote JIRA API. Throws when the API calls fail."
  [api-client project]
  (let [url (format "/project/%s" project)]
    (get-data api-client url {})))

(defn fetch-issue-types-in-project
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client project]
  (let [url (format "/project/%s" project)]
    (get-data api-client url {})))

(defn fetch-issue-types
  "Fetches issue types from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/issuetype" {}))

(defn fetch-fields
  "Fetches fields from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/field" {}))

(defn fetch-labels
  "Fetches labels from the remote JIRA API. Throws when the API calls fail.
(Jira Server has no API for getting a list of labels?!)"
  [api-client]
  (get-paged-data api-client  "/label" {:data-field :values}))

(defn fetch-statuses
  "Fetches statuses from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/status" {}))

(defn- join-double-quoted
  [separator labels]
  (string/join separator (map #(format "\"%s\"" %) labels)))

(defn fetch-projects-for-cloud [api-client]
  (get-paged-data api-client "/project/search" {:data-field :values
                                                :query-params {:jql ""}}))

(defn fetch-projects-for-server [api-client]
  (get-data api-client "/project" {:jql ""}))

(defn- in-condition [name coll]
  (format "%s in (%s)" name (join-double-quoted  "," coll)))

(defn- updated-condition [since]
  ;; Jira apparently only supports dates in jql filters 
  (format "updated> \"%s \"" (dates/date->string since)))

(defn- not-empty-coll [coll]
  (not (and (coll? coll) (empty? coll))))

(defn ->jql
  [{:keys [projects issue-types labels issues since] :as _search-options}]
    ;; JIRA doesn't allow querys like 'projects in ()' -> filter here
  (when (every? not-empty-coll [projects issue-types issues labels])
    (let [conditions (filter some?
                             [(when (seq projects) (in-condition "project" projects))
                              (when (seq issue-types) (in-condition "issuetype" issue-types))
                              (when (seq labels) (in-condition "labels" labels))
                              (when (seq issues) (in-condition "key" issues))
                              (when (some? since) (updated-condition since))])]
      (str (string/join " AND " conditions)
           " order by updated desc"))))
      
(defn search-server
  "Fetches issues from the remote JIRA API. Returns nil when the API calls fail."
  [api-client options search-options result-options]
  (let [{:keys [page-size]} options
        {:keys [fields changelog transformer] :or {transformer identity}} result-options]
    (if-let [jql (->jql search-options)]
      (get-paged-data api-client "/search"
                      {:data-field :issues
                       :page-size page-size
                       :retry? true
                       :query-params {:jql jql
                                      :fields fields
                                      :expand (when changelog "changelog")}
                       :transformer transformer})
    
      [])))

(defn fetch-changelog [api-client issue]
  ;; log duration at INFO level because we don't expect many of these calls
  ;; For most issues, the search api changelog limit of 100 should be enough to return all the transitions
  ;; and thus this call will be skipped
  (u.log/log-time (get-paged-data api-client
                                  (format "/issue/%s/changelog" issue)
                                  {:data-field :values})
                  :info "Fetch change log for Jira issue %s." issue))

(defn- approximate-count [api-client jql]
  (:count (post-data api-client "/search/approximate-count" {:jql jql})))

(defn fetch-myself
  "Fetches info about the authenticated user from the remote JIRA API. Throws when the API calls fail."
  [api-client]
  (get-data api-client "/myself" {}))

(defn search-cloud
  "Fetches issues from the remote JIRA API. Returns nil when the API calls fail."
  [api-client options search-options result-options]
  ;; The new search endpoint does not seem to return an error when authentication fails
  ;; - just an empty result. Let's call another endpoint first to make sure we get an error 
  (fetch-myself api-client)
  (let [{:keys [page-size]} options
        {:keys [projects]} search-options
        {:keys [fields changelog transformer] :or {transformer identity}} result-options]
    (if-let [jql (->jql search-options)]
      (do
        (log/infof "Fetch ~%d Jira issues from projects %s" 
                   (approximate-count api-client jql) projects)
        (get-paged-data-new api-client "/search/jql"
                            {:data-field :issues
                             :page-size page-size
                             :retry? true
                             :query-params {:jql jql
                                            :fields fields
                                            :expand (when changelog "changelog")}
                             :transformer transformer}))

      [])))

(defn jira-auth
  [username password]
  (cond
    (and (seq username) (seq password)) [username password]
    (seq password) password
    :else nil))

(comment

  ;; Clojure jira
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://clojure.atlassian.net"}
                   ;; Clojure jira allows anonymous access
                   nil))
  (fetch-changelog api-client "CLJ-2872")

  ;; CodeScene's testing jira instance
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://empear.atlassian.net"}
                   (jira-auth (System/getenv "JIRA_USER") (System/getenv "JIRA_PASSWORD"))))
  (def project "ENTERPRISE")

  ;; Apache's jira (Zookeeper)
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://issues.apache.org/jira"}
                   (jira-auth "" "")))
  (def project "ZOOKEEPER")

  ;; local Jira Server
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "http://localhost:8080"}
                   (jira-auth "" (System/getenv "JIRA_SERVER_TOKEN"))))

  (->> (search-server api-client
               {}
               {:projects [project]
                     ;:issue-types ["Story"]
                :since (dates/string->date "2025-08-01")
                :labels nil}
               {:fields ["created" "updated" "labels" "issuetype" "status"]
                :changelog true})
       
       #_(map (comp :updated :fields))
       )
  (->> (fetch-changelog api-client "ZOOKEEPER-4944")
       (map :created))
  *e
   (->> (search-cloud api-client
          {:page-size 1}
          {:projects [project]
                       ;:issue-types ["Story"]
           ;:issues ["ENTERPRISE-8"]
           :since (dates/string->date "2025-08-01")
           :labels nil}
          {:fields ["created" "updated" "labels" "issuetype" "status"]
           :changelog true
           })
        ;
        )
  (fetch-project api-client project)
  (fetch-issue-types-in-project api-client project)
  (fetch-issue-types api-client)
  (fetch-fields api-client)
  (fetch-labels api-client)
  (fetch-statuses api-client)
  (map :name (fetch-projects-for-cloud api-client))
  (map :name (fetch-projects-for-server api-client))
  (fetch-myself api-client))
