(ns codescene.features.pm-data.extractor-common
  (:require [clojure.string :as str]
            [codescene.features.pm-data.pr-data-provider :as pr-data-provider]))

(defn make-pr-ticket-ids-lookup
  [{:keys [pr-data-provider] :as _provider-def} extract-ticket-ids-fn]
  (if pr-data-provider
    (let [pull-requests (pr-data-provider/get-pull-requests pr-data-provider nil)]
      (reduce
       (fn [lookup [commit ticket-id]] (update lookup commit (fnil conj []) ticket-id))
       {}
       (for [{:keys [branch title text commits merge-commit] :as _pr} pull-requests
             ticket-id (extract-ticket-ids-fn (str/join \newline [branch title text]))
             ;; Include the merge commit to handle 'Squash and Merge'
             commit (cond-> commits merge-commit (conj merge-commit))]
         ;; TODO: Use entire hash
         [(subs commit 0 7) ticket-id])))
    {}))

(defn aliases-from 
  "Returns a map for resolving project keys using aliases.
   Note that mapping are uppercased"
  [{:keys [project-aliases] :as _provider-def}]
  (->> project-aliases
       (map (fn [[alias name]]
              [(str/upper-case alias) name]))  ;; Use upper-case when looking up
       (into {})))

(defn make-ticket-filter
  "Create a fn that filters valid ticket-ids"
  [external-project-ids aliases]
  (let [is-valid? (->> (concat external-project-ids (keys aliases))
                       (map str/upper-case)
                       set)] ;; Match actual projects AND aliases
    (fn [{:keys [scope] :as _ticket-id}]
      (-> scope str/upper-case is-valid?))))

(defn with-resolved-aliases 
  "Resolve a ticket-id match using the proved aliases.
   Note that keys are looked up in upper case"
  [aliases {:keys [scope] :as ticket-id}]
  (let [k (str/upper-case scope)
        resolved (get aliases k scope)]
    (assoc ticket-id :scope resolved)))