(ns codescene.features.delta.coverage.pr-check
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.components.db :as db.comp]
            [codescene.features.code-coverage.check.check-db :as db]
            [codescene.features.delta.coverage.cov-config :as cov]
            [codescene.features.delta.integration.integration-db :as integration-db]
            [codescene.features.util.template :as template]
            [codescene.features.project.core :as project.core]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.scheduling.persistent :as persistent]
            [codescene.features.util.maps :refer [->db-name]]
            [codescene.url.url-utils :as url]
            [taoensso.timbre :as log]
            [twarc.core :as twarc])
  (:import (java.util Date)))

(def result-coord db/result-coord)
(def insert-pr-check db/insert-check)

(defn conf->gates
  "Transforms configuration properties to a format used by CLI"
  [{:keys [coverage-metric overall-threshold changed-threshold
           overall-coverage? new-and-changed-code?] :as conf}]
  (let [->snake (comp name ->db-name)
        metric (->snake coverage-metric)]
    [{:name "overall_coverage" :threshold overall-threshold :coverage_metric metric
      :enabled overall-coverage?}
     {:name "new_and_changed_code" :threshold changed-threshold :coverage_metric metric
      :enabled new-and-changed-code?}]))

(defmulti post-cov-result
          (fn [PrIntegrationConfig _repo _result _metadata]
            (pi/-type PrIntegrationConfig)))

(defn post-cov-result*
  [tx ProjectConfiguration result metadata]
  (let [repos (project.core/-repositories ProjectConfiguration tx)
        selected-repo (some #(when (url/repo-info=repo-id % (:repo-id result)) %) repos)
        pi-config (project.core/-pr-integration-config ProjectConfiguration tx)]
    (post-cov-result pi-config selected-repo result metadata)))

(def gate-names
  {"overall_coverage" "Overall Coverage"
   "new_and_changed_code" "New & Changed Code Coverage"})

(defn awaiting-result-for-id!
  "Given an ID it will return the coverage result row, if it is awaiting
  results, and it will mark it as resolved, otherwise it will return nil"
  [db-spec id]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [result (db/get-by-id tx id)]
      (when (and result (not (:coverage-results result)))
        (log/info "Found code coverage result for %s" (result-coord result))
        (db/update-with-results tx (:project-id result) (assoc result :coverage-results {}))
        result))))

(defn post-result [tx ProjectConfiguration {:keys [coverage-results base-ref commit-sha repo-id] :as result}]
  (let [project-id (project.core/-id ProjectConfiguration)]
    (if-let [item (db/get-by-props tx project-id result)]
      (do (log/infof "Found check %s for project-id=%s, repo-id=%s, sha=%s, base-ref=%s"
                     (:id item) project-id repo-id commit-sha base-ref)
          (post-cov-result* tx
                            ProjectConfiguration
                            (merge (result-coord item)
                                   {:coverage-results coverage-results})
                            (:metadata item)))
      (let [msg (format "Couldn't find check to update for project-id=%s, repo-id=%s, sha=%s, base-ref=%s"
                        project-id repo-id commit-sha base-ref)]
        (log/info msg)
        (throw (ex-info msg {:type :not-found}))))))

(def timeout-gate
  {:pass false
   :measured-coverage nil
   :details {:pass-or-fail-reason "No valid coverage report found in the build pipeline"
             :action "Check your coverage configuration in the pipeline. CodeScene's CLI tools need to be given a valid code coverage report."}})

(defn post-timeout
  "Post result as a timeout if no result has been posted. "
  [db-spec check-id ProjectConfiguration gates]
  (if-let [{:keys [project-id metadata] :as result} (awaiting-result-for-id! db-spec check-id)]
    (let [result-per-gate (mapv #(merge % timeout-gate) gates)]
      (log/infof "PR Coverage posting timeout result for check %s on project %s" check-id project-id)
      (post-cov-result*
        db-spec
        ProjectConfiguration
        (merge result
               {:coverage-results {:all-gates-pass false
                                   :result-per-gate result-per-gate}})
        metadata))
    (log/debugf "PR Coverage check %s is already done" check-id)))

(defn iconset [provider-id]
  (let [url #(format "[![](https://codescene.io/imgs/svg/%s)](#)" %)]
    (case provider-id
      :github {:h2-pass "✅" :pass (url "pass1.svg") :fail (url "x1.svg") :h2-fail "❌"}
      (:gitlab :azure) {:h2-pass (url "pass.svg") :pass (url "pass1.svg") :fail (url "x1.svg") :h2-fail (url "x.svg")}
      {:h2-pass "✅" :pass "✅" :fail "❌" :h2-fail "❌"})))

(defn provider-options [provider-id]
  {:bottom-hr (case provider-id
                :azure "<hr>"
                :bitbucket "\n"
                "##      ")
   :icons (iconset provider-id)
   :collapsible (if (#{:bitbucket :gerrit} provider-id) false true)
   :extra-newline (if (#{:bitbucket :gitlab :azure} provider-id) "\n" "")})


(defn check-status [coverage-results]
  (let [enabled (filterv :enabled (:result-per-gate coverage-results))]
    (cond
      (empty? enabled) :disabled
      (every? :pass enabled) :success
      :else :fail)))

(defn- render [{:keys [name threshold details measured-coverage]}]
  (let [action (:action details)]
    {:gate-name (gate-names name name)
     :desc (format "required = %d%%" threshold)
     :stat (if measured-coverage (str measured-coverage "%") "-")
     :reason (:pass-or-fail-reason details)
     :action (cond
               (= "-" action) nil
               (string? action) {:description action}
               :else action)}))

(defn result->renderable
  "Process result into a data structure that fits our rendering."
  [{:keys [result-per-gate] :as res}]
  (let [{fail false pass true} (->> result-per-gate
                                    (filterv :enabled)
                                    (group-by :pass))]
    {:qg-meta {:empty-results? false :status (check-status res)}
     :qg-summary {:fail (mapv render fail)
                  :pass (mapv render pass)}}))

(defn title [renderable provider-id]
  (if (= :fail (get-in renderable [:qg-meta :status]))
    (format "%s Code Coverage Gate Failed" (:h2-fail (iconset provider-id)))
    (format "%s Code Coverage Gate Passed" (:h2-pass (iconset provider-id)))))

(defn summary
  "Provider settings are bottom-hr collapsible ex"
  [renderable provider-id]
  (template/render-file "templates/coverage/summary.md"
                        (merge renderable (provider-options provider-id))))

(def timeout-title "PR Coverage Gates Timed Out")

(defn timeout-summary
  []
  "CodeScene PR Coverage check has timed out waiting for results. Check your integration that calculates the coverage.")

(defn itemized [renderable provider-id]
  (template/render-file "templates/coverage/itemized.md"
                        (merge renderable (provider-options provider-id))))

(defn create-message [renderable provider-id]
  (str (title renderable provider-id) "\n"
       (-> (provider-options provider-id) :bottom-hr) "\n"
       (summary renderable provider-id) "\n\n"
       (itemized renderable provider-id)))

(twarc/defjob close-expired-check
  [_ project-id check-id system-sym]
  (log/infof "Closing expired PR Coverage check for project-id=%s and check-id=%s" project-id check-id)
  ;; this roundabout way of passing the system is because we can only pass
  ;; serializable data to the job, we cannot pass live objects
  (let [system @(find-var system-sym)
        ProjectConfiguration (project.core/component system project-id)]
    (jdbc/with-db-transaction [tx (db.comp/db-spec system)]
      (let [pr-config (project.core/-pr-integration-config ProjectConfiguration tx)
            conf (pi/-config-set pr-config)]
        (post-timeout tx check-id ProjectConfiguration (conf->gates (:data conf)))))))

(defn insert-pr-coverage-check
  "insert pr-coverage
      data is a map with: commit-sha, base-ref, coverage-results, repo-id
      metadata is a map with: review-id, pr-id, pr-authors, author, check-run-url, app-installation-id, pull-request-url"
  [tx {:keys [project-id system-var timeout-ms]} data metadata]
  (log/infof "Create Coverage check run for project-id=%s with data=%s and metadata=%s"
             project-id data metadata)
  (let [check-id (db/insert-check tx project-id data metadata)]
    (close-expired-check
      (persistent/scheduler)
      [project-id check-id (symbol system-var)]
      :job {:group "cov-check-close" :identity (str check-id)}
      :replace true
      :trigger {:start-at (Date. ^long (+ (System/currentTimeMillis) timeout-ms))})))

(defn repos-with-delta-other-projects
  "Finds all repos used in the project and returns repo URLs of
  repos that are used in other coverage delta analysis enabled projects."
  [tx project-id]
  (keep :repo_url (integration-db/elsewhere-cov-enabled-repos tx {:project_id project-id})))

(defn cloud-project-config
  [tx project-id]
  (integration-db/get-config tx project-id cov/cloud-properties))

(defn check-enabled?
  ([tx project-id]
   (true? (:cov-enabled? (integration-db/get-core-config tx project-id))))
  ([config-set]
   (-> config-set :metadata :cov-enabled? true?)))