(ns codescene.features.delta.coverage.cov-config
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [codescene.features.delta.integration.integration-config :as ic]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [codescene.features.api.spec.code-coverage :as cc-spec]
            [codescene.features.config.properties :refer [enum bool sval property ival] :as p]))

(s/def ::cov-qg-preset #{"global" "minimal" "medium" "all" "custom" "custom-file"})
(s/def ::initialize-check-run #{"always" "on-post"})

(def properties
  {:github [(property :initialize-check-run :cov-initialize-check-run (enum ::initialize-check-run) "always")]
   :azure []
   :bitbucket []
   :gitlab []})

(def qg-properties [(property :cov-qg-preset :cov-qg-preset (enum ::cov-qg-preset) "global")
                    (property :overall-coverage? :qg-overall-coverage (bool) false)
                    (property :new-and-changed-code? :qg-new-and-changed-code (bool) true)
                    (property :no-decline-in-coverage? :qg-no-decline-in-coverage (bool) false)
                    (property :new-file-coverage? :qg-new-file-coverage (bool) false)
                    (property :hotspot-must-improve? :qg-hotspot-must-improve (bool) false)
                    (property :modified-must-improve? :qg-modified-must-improve (bool) false)
                    (property :hotspot-coverage-target? :qg-hotspot-coverage-target (bool) false)])

(def settings-properties
  (into [(property :coverage-metric :cov-coverage-metric (enum ::cc-spec/metric) "line-coverage")
         (property :overall-threshold :cov-overall-threshold (ival) 80)
         (property :changed-threshold :cov-changed-threshold (ival) 95)]
        qg-properties))

(def qg-properties-keys (mapv :id qg-properties))
(def qg-properties-keys-wo-preset (next qg-properties-keys))

(defn profile [props]
  (let [profile-name (:cov-qg-preset props)
        all-false (zipmap qg-properties-keys-wo-preset (repeat false))
        all-true (zipmap qg-properties-keys-wo-preset (repeat true))]
    (merge props
           all-false
           (get {"minimal" {:overall-coverage? true
                            :new-and-changed-code? true}
                 "medium" {:overall-coverage? true
                           :new-and-changed-code? true
                           :no-decline-in-coverage? true
                           :new-file-coverage? true}
                 "all" all-true
                 "custom" props}
                profile-name))))

(defn merge-configuration
  [project-config global-config]
  (let [global-config (profile (:kv global-config))
        use-global? (= "global" (:cov-qg-preset project-config "global"))]
    (if use-global?
      (merge project-config (assoc global-config :cov-qg-preset "global"))
      (profile project-config))))

(def cloud-properties (update-vals properties #(into % (concat settings-properties ic/general-delta-properties))))

(defn- bitbucket-server-api-url-fix [url]
  (let [slash-trimmed (str/replace-first url #"/+$" "")]
    (if (or (= url "") (str/ends-with? slash-trimmed "/rest/api/1.0"))
      slash-trimmed (str slash-trimmed "/rest/api/1.0"))))

(def onprem-properties
  (-> (m/map-keys #(get {:github "github-app" :github-hook "github"} % (name %)) properties)
      (meta-merge
        {"github-app" [(property :secret :github-secret (sval))
                       (property :private-key :github-private-key (p/file-string true))
                       (property :app-id :github-app-id (sval))
                       (property :preset-id :github-preset-id (ival))
                       (property :api-url :github-api-url (sval) "")]
         "github" [(property :password :github-deprecated-token (sval true))
                   (property :api-url :github-deprecated-api-url (sval) "")
                   (property :host-url :github-deprecated-host-url (sval))]
         "azure" [(property :password :azure-token (sval true))
                  (property :api-url :azure-api-url (sval) "")
                  (property :preset-id :azure-preset-id (ival))
                  (property :host-url :azure-host-url (sval))]
         "bitbucket" [(property :password :bitbucket-password (sval true))
                      (property :api-url :bitbucket-api-url (sval) "")
                      (property :host-url :bitbucket-host-url (sval))
                      (property :preset-id :bitbucket-preset-id (ival))
                      (property :username :bitbucket-username (sval))]
         "bitbucket-server" [(property :build? :bitbucket-server-build (bool) true)
                             (property :report? :bitbucket-server-report (bool) true)
                             (property :annotations? :bitbucket-server-annotations (bool) true)
                             (property :password :bitbucket-server-token (sval true))
                             (assoc (property :api-url :bitbucket-server-api-url (sval) "")
                               :fix-fn bitbucket-server-api-url-fix)
                             (property :preset-id :bitbucket-server-preset-id (ival))
                             (property :host-url :bitbucket-server-host-url (sval))]
         "gerrit" [(property :password :gerrit-password (sval true))
                   (property :api-url :gerrit-api-url (sval) "")
                   (property :username :gerrit-username (sval))
                   (property :comments? :gerrit-comments (bool) false)
                   (property :origin-url :gerrit-origin-url (sval))
                   (property :host-url :gerrit-host-url (sval))
                   (property :preset-id :gerrit-preset-id (ival))
                   (property :user-hook? :gerrit-user-hook (bool) false)]
         "plugin-gerrit" {}
         "gitlab" [(property :password :gitlab-token (sval true))
                   (property :api-url :gitlab-api-url (sval) "")
                   (property :host-url :gitlab-host-url (sval))
                   (property :preset-id :gitlab-preset-id (ival))]})))

(defn from-db [props data] (p/from-db props data))